/*
 * Copyright 2007 Accu-Time Systems, Inc.  All rights reserved.
 *
 *                            DISCLAIMER
 * NO WARRANTIES
 * Accu-Time Systems expressly disclaims any warranty for the SOFTWARE
 * PRODUCT. The SOFTWARE PRODUCT and any related documentation is
 * provided "as is" without warranty of any kind, either expressed or
 * implied, including, without limitation, the implied warranties or
 * merchantability, fitness for a particular purpose, or noninfringe-
 * ment. The entire risk arising out of the use or performance of the
 * SOFTWARE PRODUCT remains with the user.
 *
 * NO LIABILITY FOR DAMAGES.
 * Under no circumstances is Accu-Time Systems liable for any damages
 * whatsoever (including, without limitation, damages for loss of busi-
 * ness profits, business interruption, loss of business information,
 * or any other pecuniary loss) arising out of the use of or inability
 * to use this product.
 *
 */

/**
 * @file BadgeEvent.java
 *
 * @brief This file contains the com.accu_time.io.BadgeEvent class
 *
 * @author Jeremy Slater - jslater@accu-time.com
 *
 * @date April 23, 2009 - initial version
 * @date May 14, 2009 - Store all event related date (port, type)
 */

package com.accu_time.device;

import java.util.EventObject;

/**
 * @class BadgeEvent
 *
 * @brief The BadgeEvent class extends the EventObject class and provides a 
 *        mechanism to generate Badge events.
 */
public class BadgeEvent extends EventObject {
	private int _port;
	private int _type;
	private Badge _badge;

	public interface Medium {
		public static final int OPTICAL = 0;
		public static final int WIEGAND = 1;
		public static final int MAGNETIC = 2;
		public static final int INVALID = 3;
	}

	/**
	 * @brief Create a BadgeEvent
	 * 
	 * The BadgeEvent constructor creates a new event container for Badge 
	 * objects originating from various device mediums.  The source port, @a 
	 * port, is a value between 0 and 3 identifying the physical device 
	 * interface.  The badge medium, @a type, identifies the type of badge 
	 * input and is defined by the BadgeEvent.Medium interface.
	 *
	 * @param source - The calling object
	 * @param badge - The Badge
	 * @param port - The physical data port.
	 * @param type - The badge medium.
	 */
	public BadgeEvent(Object source, Badge badge, int port, int type) {
		super(source);
		_port = port;
		_type = type;
		_badge = badge;
	}
	
	/**
	 * @brief Create a BadgeEvent based on the character representation of the 
	 *        medium type.
	 *
	 * This constructor generates a new BadgeEvent similar to the standard 
	 * constructor but takes a char parameter @a type representing the medium 
	 * as a single character from one of the following:
	 *  - 'O': Optical
	 *  - 'W': Wiegand, proximity badges
	 *  - 'M': Magnetic
	 *  - 'X': Invalid read
	 *
	 * @param source - The calling object
	 * @param badge - The Badge
	 * @param port - The physical data port.
	 * @param type - The badge medium.
	 */
	public BadgeEvent(Object source, Badge badge, int port, char type) {
		super(source);
		_port = port;
		
		if (type == 'O') 
			_type = Medium.OPTICAL;
		else if (type == 'W')
			_type = Medium.WIEGAND;
		else if (type == 'M')
			_type = Medium.MAGNETIC;
		else if (type == 'X')
			_type = Medium.INVALID;
		
		_badge = badge;
	}
	
	/**
	 * @brief Get the source port
	 */
	public int getPort() {
		return _port;
	}

	/**
	 * @brief Get the badge medium type
	 */
	public int getType() {
		return _type;
	}

	/**
	 * @brief The Badge storage instance
	 */
	public Badge badge() {
		return _badge;
	}
}

