/*
 * Copyright 2007 Accu-Time Systems, Inc.  All rights reserved.
 *
 *                            DISCLAIMER
 * NO WARRANTIES
 * Accu-Time Systems expressly disclaims any warranty for the SOFTWARE
 * PRODUCT. The SOFTWARE PRODUCT and any related documentation is
 * provided "as is" without warranty of any kind, either expressed or
 * implied, including, without limitation, the implied warranties or
 * merchantability, fitness for a particular purpose, or noninfringe-
 * ment. The entire risk arising out of the use or performance of the
 * SOFTWARE PRODUCT remains with the user.
 *
 * NO LIABILITY FOR DAMAGES.
 * Under no circumstances is Accu-Time Systems liable for any damages
 * whatsoever (including, without limitation, damages for loss of busi-
 * ness profits, business interruption, loss of business information,
 * or any other pecuniary loss) arising out of the use of or inability
 * to use this product.
 *
 */

/**
 * @file BadgeReader.java
 *
 * @brief This file contains the com.accu_time.io.BadgeReader class.
 *
 * @author Jeremy Slater - jslater@accu-time.com
 *
 * @date April 23, 2009 - initial version
   @date November 23, 2011 MJS Accept and ignore incompletely read badge messages from driver
 */

package com.accu_time.device;

import com.accu_time.util.Dbg;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @class BadgeReader
 *
 * @brief The BadgeReader class watches an ATS barcode device for low level 
 *        badge input.
 */
public class BadgeReader extends Thread {
    private String _path;
    private List _listeners = new ArrayList();
    private boolean _shutdown = false;
    private FileOutputStream _writer = null;
    private FileInputStream _reader = null;
    
    /**
     * @brief Create a BadgeReader instance
     *
     * The BadgeReader constructor creates a new BadgeReader instance with a
     * defined path of the ATS barcode device.  This is usually /dev/BarRdr.
     *
     * @param path - The ATS barcode device path.
     */
    public BadgeReader(String path) {
        _path = path;
    }

    /**
     * @brief Start a thread waiting for badge input.
     *
     * This function will return immediately after spawning a new thread to 
     * watch the device specified by @a path in BadgeReader.BadgeReader.  The 
     * barcode device is initialized by starting all the hardware interfaces 
     * (0-3) and enabling blocking read mode.  Barcode input events are passed 
     * up from the kernel to the file device and the data is parsed into a new 
     * Badge and a BadgeEvent is generated.
     */
    public void run() {
        try {
            int data;
            int i = 0;
            byte[] buffer = new byte[256];
            
            _writer = new FileOutputStream(_path);

            _writer.write("start 0\000".getBytes("US-ASCII"));
            _writer.write("start 1\000".getBytes("US-ASCII"));
            _writer.write("start 2\000".getBytes("US-ASCII"));
            _writer.write("start 3\000".getBytes("US-ASCII"));
            _writer.write("block\000".getBytes("US-ASCII"));
            _writer.flush();
            
            _reader = new FileInputStream(_path);
            do {
                data = _reader.read();
                buffer[i++] = (byte) data;
                String packed = "(no bytes seen)";
                if (data == -1 && i == 1)
                {
                    i = 0;
                }
                else if (data == '\u0000')
                {
                    try {
                        packed            = new String(buffer, 0, i);
                        String[] unpacked = packed.split(",");
                        String raw        = "";
                        String value      = "";
                        char type         = 'X';
                        int port          = 0;
                        
                        switch (unpacked.length) {
                            case 4: 
                                raw = unpacked[3];
                            case 3: 
                                value = unpacked[2];
                            case 2: 
                                type  = unpacked[1].charAt(0);
                            case 1: 
                                port  = Integer.parseInt(unpacked[0]);
                        }

                        Badge badge = new Badge(value, raw);
                        _fireBadgeEvent(badge, port, type);
                        i = 0;
                    } catch (Exception problem)
                    {
                        String hex = "";
                        for (int j = 0; j < i; j++)
                        {
                            hex = hex + Dbg.toHex((char)buffer[j]);
                        }
                        Dbg.out("problem " + problem + " interpreting bytes from bar code reader packed='" + packed + "' hex=" + hex);
                        i = 0;
                    }
                }
            } while (_shutdown == false);
        } catch(IOException e) {
            Dbg.out("IOException e=" + e);
            /* Just move along */
        }
    }
    
    /**
     * @brief Stop the thread reading barcode data
     */
    public void shutdown() {
        /* Force shutdown and interrupt thread
         * This doesn't really work because Java IO sucks */
        _shutdown = true;
        try{
            _reader.close();
            _writer.close();
        } catch(IOException e) {
            /* We're just closing, who cares */
        }
    }

    /**
     * @brief Add a BadgeListener for BadgeEvent events.
     *
     * All the registered BadgeListener instances will be sent a BadgeEvent 
     * whenever a barcode input event is generated by the system.
     *
     * @param l - listener instance
     */
    public synchronized void addBadgeListener(BadgeListener l) {
        _listeners.add(l);
    }
    
    /**
     * @brief Remove a BadgeListener
     *
     * @param l - listener instance
     */
    public synchronized void removeBadgeListener(BadgeListener l) {
        _listeners.remove(l);
    }
    
    private synchronized void _fireBadgeEvent(Badge badge, int port, char type) {
        BadgeEvent badgeEvent = new BadgeEvent(this, badge, port, type);
        
        Iterator listeners = _listeners.iterator();
        while (listeners.hasNext()) {
            ((BadgeListener) listeners.next()).badgeReceived(badgeEvent);
        }
    }
}
