/*
+====================================================================+
|Global Terminal                                                             |
|  Written by:  Mike Scudder/MicroTools                               |
+====================================================================+
|                                                                    |
| © 2011 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/

// Basic Java GUI controlled by the keypad and bar reader
//
// Puts up 8 buttons and a text field, a clock and a list of badges
//
// Expects a screen of about 320x240 pixels
//
// 2011-11-18 MJS Created
//

package com.accu_time.test;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import com.accu_time.device.*;
import com.accu_time.util.ClockLabel;
import com.accu_time.util.TextFileDeviceIO;

public class SimpleClockGUI extends JPanel implements ActionListener, KeyListener, MouseListener, BadgeListener, AWTEventListener
{
    private static JFrame frame;
    private static BadgeReader badgeReader;
    private static TextFileDeviceIO config;
    private static TextFileDeviceIO leds;
    private SideButton F1, F2, F3, F4, F5, F6, F7, F8;
    private ClockGUIJTextField badgeField;
    private JTextArea badgeArea;
    private String[] badgeStrings;
    private JLabel fontName;
    private BadgeListModel badgeList;
    private JCheckBox br_enable;
    private ClockLabel clk;
    private int frameWidth, frameHeight;

    private class SideButton extends JButton
    {
        private final Insets buttonMargins = new Insets(1, 1, 1, 1);
        public SideButton(String label)
        {
            super(label);
            setMargin(buttonMargins);
            setVerticalTextPosition(CENTER);
            setHorizontalTextPosition(CENTER);
        }
    }

    public class ClockGUIJTextField extends JTextField
    {
        ClockGUIJTextField(String text, int width)
        {
            super(text, width);
            /*
            Action[] aa = this.getActions();
            for (int i=0; i < aa.length; i++)
            {
                System.out.println("aa["+i+"]="+aa[i]);
            }
            */
        }

        public void badgeUpdate(String text)
        {
            this.setText(text);
            this.revalidate();
        }
    }
    
    private class BadgeListModel
    {
        String[] badgeStringList = { "0000000001" }; // initially, a single badge with a 10 digit badge string with value 1
        final static int maxBadgeListLength = 10;
        private int matchingBadgeIndex(String badgeName)
        {
            for (int i=0; i < badgeStringList.length; i++)
            {
                if (badgeName.equals(badgeStringList[i]))
                {
                    return i;
                }
            }
            return -1;
        }

        public boolean badgeAccepted(String badgeName)
        {
            return (matchingBadgeIndex(badgeName) >= 0);
        }
        
        public boolean deleteBadge(String badgeName)
        {
            int index = matchingBadgeIndex(badgeName);
            if (index >= 0)
            {
                String[] newBadgeStringList = new String[badgeStringList.length - 1];
                for (int i=0; i < index; i++)
                {
                    newBadgeStringList[i] = badgeStringList[i];
                }
                for (int i=index; i < badgeStringList.length-1; i++)
                {
                    newBadgeStringList[i] = badgeStringList[i+1];
                }
                badgeStringList = newBadgeStringList;
                return true;
            }
            return false;
        }

        public String getText()
        {
            String text = "";
            if (badgeStringList.length > 0)
            {
                text = badgeStringList[0];
                for (int i = 1; i < badgeStringList.length; i++)
                {
                    text = text + "\n" + badgeStringList[i];
                }
            }
            return text;
        }

        public boolean addBadge(String badgeName)
        {
            int index = matchingBadgeIndex(badgeName);
            if (index == -1 && badgeStringList.length < maxBadgeListLength)
            {
                String[] newBadgeStringList = new String[badgeStringList.length + 1];
                for (int i=0; i < badgeStringList.length; i++)
                {
                    newBadgeStringList[i] = badgeStringList[i];
                }
                newBadgeStringList[badgeStringList.length] = badgeName;
                badgeStringList = newBadgeStringList;
                return true;
            }
            return false;
        }
    }
    
    String fontToName(Font font)
    {
        String family = font.getFamily();
        int style = font.getStyle();
        String styleName;
        int size = font.getSize();
        
        if (style == Font.PLAIN)
        {
            styleName = "PLAIN";
        }
        else if (style == Font.BOLD)
        {
            styleName = "BOLD";
        }
        else if (style == Font.ITALIC)
        {
            styleName = "ITALIC";
        }
        else if (style == Font.BOLD + Font.ITALIC)
        {
            styleName = "BOLD+ITALIC";
        }
        else
        {
            styleName = "UNKNOWN";
        }
        return family + " " + styleName + " " + size;
    }
    
    private class FontFamilyChangeAction extends AbstractAction
    {
        String[] fontFamilies;
        int fontFamilyCount;
        int currentFontFamilyIndex = 0;
        FontFamilyChangeAction()
        {
            super("Font");
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            fontFamilies = ge.getAvailableFontFamilyNames();
            fontFamilyCount = fontFamilies.length;
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Font"))
            {
                return;
            }
            currentFontFamilyIndex++;
            if (currentFontFamilyIndex >= fontFamilyCount)
            {
                currentFontFamilyIndex = 0;
            }
            //System.out.println("Changing font family to "+fontFamilies[currentFontFamilyIndex]);
            int Style = badgeField.getFont().getStyle();
            int Size = badgeField.getFont().getSize();
            Font newFont = new Font(fontFamilies[currentFontFamilyIndex], Style, Size);
            badgeField.setFont(newFont);
            badgeArea.setFont(newFont);
            fontName.setText(fontToName(newFont));
        }
    }
    
    private class FontStyleChangeAction extends AbstractAction
    {
        String[] fontStyles = new String[] { "PLAIN", "BOLD", "ITALIC", "BOLD+ITALIC" };
        int[] fontStyleConstants = new int[] { Font.PLAIN, Font.BOLD, Font.ITALIC, Font.BOLD+Font.ITALIC };
        int fontStyleCount = fontStyles.length;
        int currentFontStyleIndex = 0;
        FontStyleChangeAction()
        {
            super("Style");
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Style"))
            {
                return;
            }
            currentFontStyleIndex++;
            if (currentFontStyleIndex >= fontStyleCount)
            {
                currentFontStyleIndex = 0;
            }
            //System.out.println("Changing font style to "+fontStyles[currentFontStyleIndex]);
            String Family = badgeField.getFont().getFamily();
            int Size = badgeField.getFont().getSize();
            Font newFont = new Font(Family, fontStyleConstants[currentFontStyleIndex], Size);
            badgeField.setFont(newFont);
            badgeArea.setFont(newFont);
            fontName.setText(fontToName(newFont));
        }
    }
    
    private class FontSizeChangeAction extends AbstractAction
    {
        int[] fontSizes = { 6, 8, 10, 12, 16, 24 };
        int fontSizeCount = fontSizes.length;
        int currentFontSizeIndex = 3; // Start at size 10
        FontSizeChangeAction()
        {
            super("Size");
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Size"))
            {
                return;
            }
            currentFontSizeIndex++;
            if (currentFontSizeIndex >= fontSizeCount)
            {
                currentFontSizeIndex = 0;
            }
            //System.out.println("Changing font size to "+fontSizes[currentFontSizeIndex]);
            String Family = badgeField.getFont().getFamily();
            int Style = badgeField.getFont().getStyle();
            Font newFont = new Font(Family, Style, fontSizes[currentFontSizeIndex]);
            badgeField.setFont(newFont);
            badgeArea.setFont(newFont);
            fontName.setText(fontToName(newFont));
        }
    }
    
    private class BadgeReaderEnableAction extends AbstractAction
    {
        boolean enabled = false;
        BadgeReaderEnableAction()
        {
            super("Reader");
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Reader"))
            {
                return;
            }
            enabled = !enabled;
            System.out.println("badge reader enable=" + enabled);
            br_enable.setSelected(enabled);
        }
    }
    
    private class BadgeAddAction extends AbstractAction
    {
        BadgeAddAction(String name)
        {
            super(name);
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Add"))
            {
                return;
            }
            String badge = badgeField.getText();
            badgeField.setText("");
            if (badgeList.addBadge(badge))
            {
                System.out.println("Added badge " + badge);
            }
            badgeArea.replaceRange(badgeList.getText(), 0, badgeArea.getDocument().getLength());
            badgeArea.getParent().repaint();
        }
    }
    
    private class BadgeDeleteAction extends AbstractAction
    {
        BadgeDeleteAction(String name)
        {
            super(name);
        }
        
        public void actionPerformed(ActionEvent e)
        {
            if (!e.getActionCommand().equals("Delete"))
            {
                return;
            }
            String badge = badgeField.getText();
            badgeField.setText("");
            if (badgeList.deleteBadge(badge))
            {
                System.out.println("Deleted badge " + badge);
            }
            badgeArea.replaceRange(badgeList.getText(), 0, badgeArea.getDocument().getLength());
            badgeArea.getParent().repaint();
            System.out.println("new badge area text should be:\n" + badgeList.getText());
        }
    }
    
    public SimpleClockGUI(java.awt.Rectangle bounds)
    {
        frameWidth = bounds.width;
        frameHeight = bounds.height;
        BorderLayout bl = new BorderLayout();
        this.setLayout(bl);
        GridLayout left_gl = new GridLayout(4, 1);
        JPanel leftButtons = new JPanel(left_gl);
        leftButtons.setPreferredSize(new Dimension(frameWidth / 5, frameHeight));
        GridLayout right_gl = new GridLayout(4, 1);
        JPanel rightButtons = new JPanel(right_gl);
        rightButtons.setPreferredSize(new Dimension(frameWidth / 5, frameHeight));
        GridBagLayout cp_gb = new GridBagLayout();
        JPanel cp = new JPanel(cp_gb);

        Font startFont = new Font("SansSerif", Font.PLAIN, 10);
        setFont(startFont);
        fontName = new JLabel(fontToName(startFont), SwingConstants.CENTER);
        GridBagConstraints cfn = new GridBagConstraints();
        cfn.anchor = GridBagConstraints.NORTH;
        cfn.weightx = 1.0;
        cfn.weighty = 1.0;
        cfn.gridx = 0;
        cfn.gridy = 0;
        cp_gb.setConstraints(fontName, cfn);
        cp.add(fontName);
        fontName.setFocusable(false);
        
        badgeField = new ClockGUIJTextField("", 14);
        badgeField.addActionListener(this);
        badgeField.addKeyListener(this);
        badgeField.addMouseListener(this);
        GridBagConstraints cbf = new GridBagConstraints();
        cbf.anchor = GridBagConstraints.CENTER;
        cbf.weightx = 1.0;
        cbf.weighty = 1.0;
        cbf.gridx = 0;
        cbf.gridy = 1;
        cp_gb.setConstraints(badgeField, cbf);
        cp.add(badgeField);
        badgeField.setFocusable(true);
        badgeField.setRequestFocusEnabled(true);
        badgeField.requestFocusInWindow();
        
        badgeList = new BadgeListModel();
        badgeArea = new JTextArea(badgeList.getText(), 10, 14);
        badgeArea.setBackground(badgeArea.getBackground().darker());
        badgeArea.setEditable(false);
        GridBagConstraints cba = new GridBagConstraints();
        cba.anchor = GridBagConstraints.CENTER;
        cba.weightx = 1.0;
        cba.weighty = 10.0;
        cba.gridx = 0;
        cba.gridy = 2;
        cba.gridwidth = 1;
        cba.gridheight = 10;
        cp_gb.setConstraints(badgeArea, cba);
        cp.add(badgeArea);
        badgeArea.setFocusable(false);

        br_enable = new JCheckBox("Enable Badge Reader", false);
        br_enable.setAction(new BadgeReaderEnableAction());
        br_enable.setText("Enable Badge Reader");
        GridBagConstraints cbr = new GridBagConstraints();
        cbr.weightx = 1.0;
        cbr.weighty = 1.0;
        cbr.gridx = 0;
        cbr.gridy = 12;
        cp_gb.setConstraints(br_enable, cbr);
        cp.add(br_enable);
        // This should get focus from the keyboard but just change state from the mouse
        br_enable.setFocusable(true);
        br_enable.setRequestFocusEnabled(false);
        br_enable.addKeyListener(this);
        br_enable.addMouseListener(this);
        
        clk = new ClockLabel();
        GridBagConstraints cclk = new GridBagConstraints();
        //cclk.anchor = GridBagConstraints.SOUTH;
        cclk.weightx = 1.0;
        cclk.weighty = 1.0;
        cclk.gridx = 0;
        cclk.gridy = 13;
        cp_gb.setConstraints(clk, cclk);
        cp.add(clk);
        clk.setFocusable(false);
        
        cp.setPreferredSize(new Dimension(3 * frameWidth / 5, frameHeight));

        F1 = addButton(left_gl, SwingConstants.WEST, 0, 'a', "Font", new FontFamilyChangeAction(), this);
        F1.setIconTextGap(1);
        leftButtons.add(F1);
        F2 = addButton(left_gl, SwingConstants.WEST, 1, 'b', "Size", new FontStyleChangeAction(), this);
        F2.setIconTextGap(1);
        leftButtons.add(F2);
        F3 = addButton(left_gl, SwingConstants.WEST, 2, 'c', "Style", new FontSizeChangeAction(), this);
        F3.setIconTextGap(1);
        leftButtons.add(F3);
        F4 = addButton(left_gl, SwingConstants.WEST, 3, 'd', "Reader", new BadgeReaderEnableAction(), this);
        F4.setIconTextGap(1);
        leftButtons.add(F4);

        F5 = addButton(right_gl, SwingConstants.EAST, 0, 'e', "Add", new BadgeAddAction("Add"), this);
        rightButtons.add(F5);
        F6 = addButton(right_gl, SwingConstants.EAST, 1, 'f', "Delete", new BadgeDeleteAction("Delete"), this);
        rightButtons.add(F6);
        F7 = addButton(right_gl, SwingConstants.EAST, 2, 'g', "Choose", null, this);
        rightButtons.add(F7);
        F8 = addButton(right_gl, SwingConstants.EAST, 3, 'h', "Exit", null, this);
        rightButtons.add(F8);

        add(leftButtons, BorderLayout.WEST);
        add(cp, BorderLayout.CENTER);
        add(rightButtons, BorderLayout.EAST);
        this.validate();
    }

    private SideButton addButton(GridLayout gbl, int side, int row, char key, String label, AbstractAction action, ActionListener listener)
    {
        SideButton b = new SideButton(label);
        if (action != null)
        {
            b.setAction(action);
        }
        b.addActionListener(listener);
        return b;
    }

    public final int CENTER_KEY = 32;
    public final int F1_KEY = 65;
    public final int F2_KEY = 66;
    public final int F3_KEY = 67;
    public final int F4_KEY = 68;
    public final int F5_KEY = 69;
    public final int F6_KEY = 70;
    public final int F7_KEY = 71;
    public final int F8_KEY = 72;
    public void keyPressed(KeyEvent event)
    {
        int keyCode = event.getKeyCode();
        System.out.println("KeyCode="+keyCode);
        if (keyCode == F1_KEY)
        {
            event.consume();
            F1.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+1, "Font"));
        }
        else if (keyCode == F2_KEY)
        {
            event.consume();
            F2.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+2, "Style"));
        }
        else if (keyCode == F3_KEY)
        {
            event.consume();
            F3.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+3, "Size"));
        }
        else if (keyCode == F4_KEY)
        {
            event.consume();
            F4.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+4, "Reader"));
        }
        else if (keyCode == F5_KEY)
        {
            event.consume();
            F5.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+5, "Add"));
        }
        else if (keyCode == F6_KEY)
        {
            event.consume();
            F6.getAction().actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST+6, "Delete"));
        }
        else if (keyCode == F7_KEY) // equivalent to the enter key
        {
            event.consume();
            this.actionPerformed(null);
        }
        else if (keyCode == F8_KEY)
        {
            event.consume();
            System.exit(0);
        }
    }

    static Timer startupFocusTimer;
    private static class StartupFocus implements ActionListener
    {
        JComponent focus;
        StartupFocus(JComponent field)
        {
            focus = field;
        }
            
        public void actionPerformed(ActionEvent evt)
        {
            System.out.println("requesting focus for " + focus);
            focus.requestFocusInWindow();
            focus.repaint();
            startupFocusTimer.stop();
        }
    }

    public void keyReleased(KeyEvent event) {
    }

    public void keyTyped(KeyEvent event) {
        if (event.getKeyChar() >= 'a' && event.getKeyChar() <= 'h')
        {
            event.consume();
            return;
        }
        char keyChar = event.getKeyChar();
        //System.out.println("keyTyped char='"+keyChar+"' text=\""+badgeField.getText()+"\"");
        //badgeField.requestFocusInWindow();
    }

    public void mouseClicked(MouseEvent e)
    {
    }
    
    public void mouseEntered(MouseEvent e)
    {
    }
    
    public void mouseExited(MouseEvent e)
    {
    }
    
    public void mousePressed(MouseEvent e)
    {
        System.out.println("mouse button " + e.getButton() + " pressed at (" + e.getXOnScreen() + "," + e.getYOnScreen() + ")");
    }
    
    public void mouseReleased(MouseEvent e)
    {
    }
    
    public void badgeReceived(BadgeEvent event)
    {
        String text = event.badge().getValue();
        System.out.println("badge text="+text+" accepted="+br_enable.isSelected());
        if (br_enable.isSelected())
        {
            badgeField.badgeUpdate(text);
            badgeField.requestFocusInWindow();
        }
    }

    Timer removeNoticeTimer;
    private class RemoveNotice implements ActionListener
    {
        public void actionPerformed(ActionEvent evt)
        {
            badgeArea.setText(badgeList.getText());
            badgeArea.repaint();
            removeNoticeTimer.stop();
        }
    }

    public void beep()
    {
        try
        {
            java.lang.Runtime.getRuntime().exec("beep 5000 50");
        }
        catch (Exception ex)
        {
            System.out.println("beep() failed exception=" + ex);
        }
    }

    public void led(String name, int onSeconds)
    {
        // use array of strings to avoid the exec method breaking the command at spaces
        String[] cmd = {"echo", name + " = " + onSeconds, ">", "/dev/atsleds"};
        try
        {
            java.lang.Runtime.getRuntime().exec(cmd);
            System.out.println("execed " + cmd);
        }
        catch (Exception ex)
        {
            System.out.println("cmd '" + cmd + "' failed exception=" + ex);
        }
    }
    
    Timer delayedBeepTimer;
    private class DelayedBeep implements ActionListener
    {
        public void actionPerformed(ActionEvent evt)
        {
            beep();
            delayedBeepTimer.stop();
        }
    }
    
    public void double_beep()
    {
        beep();
        delayedBeepTimer = new Timer(250, new DelayedBeep());
        delayedBeepTimer.start();
    }

    public void actionPerformed(ActionEvent event)
    {
        String text = badgeField.getText();
        System.out.println("gui text="+text);
        if (badgeList.badgeAccepted(text))
        {
            System.out.println("Badge '" + text + "' Accepted");
            badgeArea.setText("Badge " + text + " accepted");
            leds.putValue("Good Led", "1");
            leds.putValue("Bad Led", "0");
            leds.write();
            beep();
        }
        else
        {
            System.out.println("Badge '" + text + "' Rejected");
            badgeArea.setText("Badge " + text + " rejected");
            leds.putValue("Good Led", "0");
            leds.putValue("Bad Led", "1");
            leds.write();
            double_beep();
        }
        badgeArea.repaint();
        if (badgeField.getText().equals("54321")) { System.exit(1); }
        badgeField.setText("");
        badgeField.repaint();
        badgeField.requestFocusInWindow();
        removeNoticeTimer = new Timer(1000 * 5, new RemoveNotice());
        removeNoticeTimer.start();
        //JOptionPane.showMessageDialog(badgeArea, );
    }

    public void eventDispatched(AWTEvent event)
    {
        System.out.println("saw AWTEvent " + event.toString());
    }

    private static void transparent_cursor(JFrame frame)
    {
        int[] cursor_pixels = new int [16*16];
        java.awt.Image image = java.awt.Toolkit.getDefaultToolkit().createImage(new java.awt.image.MemoryImageSource(16, 16, cursor_pixels, 0, 16));
        java.awt.Cursor transparent_cursor = java.awt.Toolkit.getDefaultToolkit().createCustomCursor(image, new java.awt.Point(0,0), "transparentCursor");
        frame.setCursor(transparent_cursor);
    }

    private static void listInfo()
    {
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        String[] fontFamilies = ge.getAvailableFontFamilyNames();
        for (int i = 0; i < fontFamilies.length; i++)
        {
            System.out.println("f["+i+"]="+fontFamilies[i]);
        }
        java.awt.GraphicsDevice[] gdevs = ge.getScreenDevices();
        for (int i = 0; i < gdevs.length; i++)
        {
            System.out.println("gd["+i+"]="+gdevs[i]);
            System.out.println("accelerated="+gdevs[i].getAvailableAcceleratedMemory()+" id="+gdevs[i].getIDstring()+" type="+gdevs[i].getType());
            java.awt.GraphicsConfiguration[] gc = gdevs[i].getConfigurations();
            for (int j = 0; j < gc.length; j++)
            {
                System.out.println("gc["+j+"]="+gc[j]);
            }
            java.awt.DisplayMode[] dm = gdevs[i].getDisplayModes();
            for (int j = 0; j < dm.length; j++)
            {
                System.out.println("dm["+j+"]="+dm[j]+" width,height,depth="+dm[j].getWidth()+","+dm[j].getHeight()+","+dm[j].getBitDepth()+" refresh_rate="+dm[j].getRefreshRate());
            }
        }
        System.out.println("center="+ge.getCenterPoint().x+","+ge.getCenterPoint().y+" size="+ge.getMaximumWindowBounds().width+","+ge.getMaximumWindowBounds().height);
        System.out.println("JFrame size="+frame.getSize());
        System.out.println("Mouse buttons="+java.awt.MouseInfo.getNumberOfButtons()+" at ("+java.awt.MouseInfo.getPointerInfo().getLocation()+") dev="+java.awt.MouseInfo.getPointerInfo().getDevice());
    }

    public static SimpleClockGUI makeGUI()
    {
        final BadgeReader badgeReader = new BadgeReader("/dev/BarRdr");
        java.awt.Rectangle default_bounds = GraphicsEnvironment.getLocalGraphicsEnvironment().getMaximumWindowBounds();
        System.out.println("default_bounds="+default_bounds);
        frame = new JFrame("SimpleClockGUI");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        final SimpleClockGUI gui = new SimpleClockGUI(default_bounds);
        final long mask = AWTEvent.INPUT_METHOD_EVENT_MASK + AWTEvent.KEY_EVENT_MASK + AWTEvent.CONTAINER_EVENT_MASK + AWTEvent.ACTION_EVENT_MASK +
                          AWTEvent.MOUSE_EVENT_MASK + AWTEvent.FOCUS_EVENT_MASK + AWTEvent.TEXT_EVENT_MASK + AWTEvent.WINDOW_EVENT_MASK + AWTEvent.WINDOW_FOCUS_EVENT_MASK;
        Toolkit.getDefaultToolkit().addAWTEventListener(gui, mask);
        final BadgeMiddleMan bmm = new BadgeMiddleMan(gui);
        badgeReader.addBadgeListener(bmm);
        frame.addKeyListener(gui);
        frame.add(gui);
        frame.setSize(default_bounds.width, default_bounds.height);
        frame.addWindowFocusListener(new WindowAdapter() { public void WindowGainedFocus(WindowEvent e) { System.out.println("frame gained focus e="+e); gui.badgeField.requestFocusInWindow();}});
        gui.setPreferredSize(frame.getSize());
        frame.pack();
        transparent_cursor(frame);
        //gui.badgeField.requestFocusInWindow();
        frame.setVisible(true);
        frame.toFront();
        //gui.badgeField.grabFocus();
        java.awt.Toolkit.getDefaultToolkit().beep();
        System.out.println("frame size = (" + frame.getSize().width + "," + frame.getSize().height + ")");
        badgeReader.start();
        StartupFocus focus = new StartupFocus(gui.badgeField);
        startupFocusTimer = new Timer(1, focus);
        startupFocusTimer.start();
        listInfo();
        return gui;
    }

    public static class BadgeMiddleMan implements BadgeListener
    {
        BadgeEvent passedEvent;
        Object lock;
        final SimpleClockGUI gui;
        final Runnable badgePass;

        
        BadgeMiddleMan()
        {
           this((SimpleClockGUI)null);
        }

        BadgeMiddleMan(SimpleClockGUI gui_to_use)
        {
            gui = gui_to_use;
            lock = new Object();
            badgePass = new Runnable(){
                public void run()
                {
                    synchronized (lock)
                    {
                        if (gui != null)
                        {
                            gui.badgeReceived(passedEvent);
                        }
                    }
                }
            };
        }

        public void badgeReceived(BadgeEvent event)
        {
            synchronized (lock)
            {
                passedEvent = new BadgeEvent(this, event.badge(), event.getPort(), event.getType());
                SwingUtilities.invokeLater(badgePass);
            }
        }
    }

    public static void main(String[] args) throws Exception
    {
        config = new TextFileDeviceIO("/dev/atsconfig");
        leds = new TextFileDeviceIO("/dev/atsleds");
        System.out.println("display_type="+config.readValue("displaytype"));
        javax.swing.SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                SimpleClockGUI gui = makeGUI();
            }
        });
    }
}
