/*
+====================================================================+
|Global Terminal                                                             |
|  Written by:  Mike Scudder/MicroTools                               |
+====================================================================+
|                                                                    |
|  2007 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/

package com.accu_time.test;

import com.accu_time.util.BiometricAPI;
import com.accu_time.util.BiometricAPI.*;
import com.accu_time.util.BiometricAPIviaJNI;
import com.accu_time.util.Dbg;
import com.accu_time.util.MQ;
import java.io.*;

public class TestJavaAPI {
	private static BufferedReader stdin = new BufferedReader( new InputStreamReader( System.in ) );

	static void loadFlashFromFile(int devType, String filename) {
		char rs = '\u001E';
		String prefix = "FPFL"+rs;
		byte [] flashRecord = null;
		IntPass record = new IntPass(0);
		IntPass done = new IntPass(0);

		BiometricAPIviaJNI bio = new BiometricAPIviaJNI();
		IntPass ip = new IntPass();
		System.out.println(bio.errString(bio.BioInitialization(ip, devType, (byte)3)) + " = BioInitialization(ip, " + devType + ", (byte)4)");
		int handle = ip.i;
		System.out.println("handle = " + handle);
		int responsive = bio.BioDeviceResponsive(handle);
		System.out.println(bio.errString(responsive) + " = bio.BioDeviceResponsive(handle)");
		if (responsive != BiometricAPI.BIOAPI_OK) {
			System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOff(handle)");
			bio.wait(2); // give it time to really power off
		}
		StringPass versionString = new StringPass();
		responsive = bio.BioGetVersionString(handle, versionString);
		System.out.println(bio.errString(responsive) + " = bio.BioGetVersionString(handle, versionString)");
		if (responsive != BiometricAPI.BIOAPI_OK) {
			System.out.println("Fingerprint device not responding");
			return;
		}
		System.out.println();
		System.out.println("Version String before Flashing = " + versionString.s);
		System.out.println();
		System.out.println("loading flash image");
		try {
			BufferedReader flashIn = new BufferedReader(new FileReader(filename));
			String line = flashIn.readLine();
			int expectedLineNumber = 0;
			while (line != null) {
				try {
					if (!line.startsWith(prefix)) {
						throw new Exception ("Ignoring line with unrecognized prefix");
					}
					int nextRsAt = line.indexOf(rs, prefix.length());
					if (nextRsAt == -1) {
						throw new Exception ("Ignoring line with only one record separator");
					}
					String embeddedLineNumberString = line.substring(prefix.length(), nextRsAt);
					int embeddedLineNumber = Integer.parseInt(embeddedLineNumberString);
					if (expectedLineNumber != embeddedLineNumber) {
						System.out.println("Unexpected line number: got " + embeddedLineNumber + " while expecting " + expectedLineNumber);
						return;
					}
					flashRecord = line.substring(prefix.length()).getBytes("US-ASCII");
					record.i = embeddedLineNumber;
					done.i = 0;
					if (embeddedLineNumber == 0 || embeddedLineNumber%10 == 0) {
						System.out.print("Flashing record " + embeddedLineNumber + " ");
					}
					int result = bio.BioFlashLoad(handle, flashRecord, record, done);
					if (result != BiometricAPI.BIOAPI_OK) {
						System.out.println("\nERROR " + bio.errString(result) + " at record " + embeddedLineNumber);
						return;
					}
					if (embeddedLineNumber == 0 || embeddedLineNumber%10 == 0) {
						System.out.println(" result = " + bio.errString(result));
					} else {
						System.out.print(".");
					}
					if (done.i == 1) {
						System.out.println(" done result = " + bio.errString(result));
						responsive = bio.BioDeviceResponsive(handle);
						System.out.println(bio.errString(responsive) + " = bio.BioDeviceResponsive(handle)");
						if (responsive != BiometricAPI.BIOAPI_OK) {
							System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOff(handle)");
							bio.wait(2); // give it time to really power off
						}
						responsive = bio.BioGetVersionString(handle, versionString);
						System.out.println(bio.errString(responsive) + " = bio.BioGetVersionString(handle, versionString)");
						if (responsive != BiometricAPI.BIOAPI_OK) {
							System.out.println("Fingerprint device not responding");
							return;
						}
						System.out.println();
						System.out.println("Version String after Flashing = " + versionString.s);
						System.out.println("Successfully flashed " + embeddedLineNumber + " records");
						return;
					}
					expectedLineNumber++;
				} catch (Exception ex) {
					System.out.println("Non fatal exception " + ex + ": skipping line:\n" + line);
				}
				line = flashIn.readLine();
			}
			flashIn.close();
		} catch (Exception ex) {
			System.out.println("Exception " + ex + " encountered");
			return;
		}
		return;
	}

    public static void main(String args[]) throws IOException, NumberFormatException {
		int errCode = BiometricAPI.BIOAPI_INVALID;
		int devType = -1;
		if (args.length >= 1 && args[0].equalsIgnoreCase("Cogent")) {
			devType = BiometricAPI.BIOAPI_COGENT;
		} else if (args.length >= 1 && args[0].equalsIgnoreCase("Bioscrypt")) {
			devType = BiometricAPI.BIOAPI_BIOSCRYPT;
		} else if (args.length >= 1 && args[0].equalsIgnoreCase("Suprema")) {
			devType = BiometricAPI.BIOAPI_SUPREMA;
		}
		if (devType == -1 ||
		    (args.length == 3 && ! args[1].equalsIgnoreCase("Flash"))) {
		    System.out.println("args length: " + args.length);
			System.out.println("Usage:\n" +
							   "sh test.sh BiometricAPIviaJNI Cogent|Bioscrypt|Suprema [flash flashFileName]");
			return;
		}
		BioscryptQuality_struct bioscryptQuality = new BioscryptQuality_struct();
		// Test all the above...
		System.loadLibrary("ATSJavaAPI");
		if (args.length == 3 && args[1].equalsIgnoreCase("Flash")) {
			loadFlashFromFile(devType, args[2]);
			return;
		}
		BiometricAPIviaJNI bio = new BiometricAPIviaJNI();
		IntPass ip = new IntPass();
		System.out.println(bio.errString(bio.BioInitialization(ip, devType, (byte)3)) + " = BioInitialization(ip, " + devType + ", (byte)4)");
		int handle = ip.i;
		System.out.println("handle = " + handle);
		System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOff(handle)");
		bio.wait(2); // give it time to really power off
		System.out.println(bio.errString(bio.BioInitComPort(handle, 57600)) + " = bio.BioInitComPort(handle, 57600)");
		errCode = bio.BioDeviceResponsive(handle);
		System.out.println(bio.errString(errCode) + " = bio.BioDeviceResponsive(handle)");
		if (errCode != BiometricAPI.BIOAPI_OK) {
			System.out.println("Couldn't talk to fingerprint reader");
			return;
		}
		
		if (devType == BiometricAPI.BIOAPI_COGENT) {
			System.out.println("Checking Threshold Setting");
			System.out.println("Reading Threshold");
			
			IntPass ip2 = new IntPass();
			
			System.out.println(bio.errString(bio.BioThreshold(handle, 0)) + " = bio.Threshold(handle, 0)");
			System.out.println(bio.errString(bio.BioGetThreshold(handle, ip2)) + " = bio.Threshold(handle, ip2)");
			System.out.println("TestJavaAPI.Java - After Call Value = " + ip2.i);
		}
		
		System.out.println("Waiting 5 seconds");
		bio.wait(5);
		BytePass bp = new BytePass();
		System.out.println();
		if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			bioscryptQuality = new BioscryptQuality_struct();
			bioscryptQuality.content = 0;
			bioscryptQuality.quality = 0;
			bioscryptQuality.contentThresholds = new byte [] {80, 60, 40};
			bioscryptQuality.qualityThresholds = new byte [] {40, 60, 80};
			System.out.println("Using content thresholds {" +
				bioscryptQuality.contentThresholds[0] + ", " +
				bioscryptQuality.contentThresholds[1] + ", " +
				bioscryptQuality.contentThresholds[2] + "}");
			System.out.println("Using quality thresholds {" +
				bioscryptQuality.qualityThresholds[0] + ", " +
				bioscryptQuality.qualityThresholds[1] + ", " +
				bioscryptQuality.qualityThresholds[2] + "}");
		}
		System.out.println("Present finger for enroll");
		System.out.println();
		// different enroll if Bioscrypt
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {
			System.out.println(bio.errString(bio.BioEnroll(handle, 12345, (byte)0, bp)) + " = bio.BioEnroll(handle, 12345, (byte)0, bp)");
			System.out.println("quality = " + bp.b);
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			System.out.println(bio.errString(bio.BioEnroll(handle, 12345, (byte)0, bioscryptQuality)) + " = bio.BioEnroll(handle, 12345, (byte)0, bioscryptQuality)");
			System.out.println("content = " + bioscryptQuality.content);
			System.out.println("quality = " + bioscryptQuality.quality);
		}
		System.out.println();
		System.out.println("remove finger");
		bio.wait(2);
		System.out.println("present finger for verify");
		System.out.println(bio.errString(bio.BioVerify(handle, 12345)) + " = bio.BioVerify(handle, 12345)");
		System.out.println();
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {		
			System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOff(handle)");
			System.out.println(bio.errString(bio.BioInitComPort(handle, 9600)) + " = bio.BioInitComPort(handle, 9600)");
			bio.wait(2); // give it time to really power off
			System.out.println(bio.errString(bio.BioPowerOn(handle)) + " = bio.BioPowerOn(handle)");
		}
		StringPass selfTestResult = new StringPass();
		System.out.println(bio.errString(bio.BioSelfTest(handle, selfTestResult)) + " = bio.BioSelfTest(handle, selfTestResult)");
		System.out.println("selfTestResult.s = " + selfTestResult.s);
		if (devType == BiometricAPI.BIOAPI_COGENT) {		
			System.out.println(bio.errString(bio.BioInitComPort(handle, 19200)) + " = bio.BioInitComPort(handle, 19200)");
			System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOn(handle)");
			bio.wait(2); // give it time to really power off
		}
		StringPass versionString = new StringPass();
		System.out.println(bio.errString(bio.BioGetVersionString(handle, versionString)) + " = bio.BioGetVersionString(handle, versionString)");
		System.out.println("versionString.s = " + versionString.s);
		if (devType == BiometricAPI.BIOAPI_COGENT) {		
			System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOn(handle)");
			System.out.println(bio.errString(bio.BioInitComPort(handle, 38400)) + " = bio.BioInitComPort(handle, 38400)");
			bio.wait(2); // give it time to really power off
		}
		System.out.println(bio.errString(bio.BioDeviceResponsive(handle)) + " = bio.BioDeviceResponsive(handle)");
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {		
			System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOn(handle)");
			System.out.println(bio.errString(bio.BioInitComPort(handle, 115200)) + " = bio.BioInitComPort(handle, 115200)");
			bio.wait(2); // give it time to really power off
		}
		IntPass subType = new IntPass(-1);
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {
			System.out.println("expect NOT_SUPPORTED");
		}
		System.out.println(bio.errString(bio.BioGetDeviceSubType(handle, subType)) + " = bio.BioGetDeviceSubType(handle, subType)");
		System.out.println("subType.i = " + subType.i);
		System.out.println(bio.errString(bio.BioPowerOff(handle)) + " = bio.BioPowerOn(handle)");
		System.out.println(bio.errString(bio.BioInitComPort(handle, 57600)) + " = bio.BioInitComPort(handle, 57600)");
		bio.wait(2); // give it time to really power off
		System.out.println();
		System.out.println("Enroll a secondary finger for badge 12345");
		System.out.println();
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {		
			System.out.println(bio.errString(bio.BioEnroll(handle, 12345, (byte)1, bp)) + " = bio.BioEnroll(handle, 12345, (byte)0, bp)");
			System.out.println("quality = " + bp.b);
		} if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			System.out.println(bio.errString(bio.BioEnroll(handle, 12345, (byte)1, bioscryptQuality)) + " = bio.BioEnroll(handle, " + 12345 + ", (byte)1, bioscryptQuality)");
			System.out.println("content = " + bioscryptQuality.content);
			System.out.println("quality = " + bioscryptQuality.quality);
		}
		System.out.println();
		System.out.println("Choose another badge number:");
		String fromConsole = stdin.readLine();
		int badgeNumber = Integer.parseInt(fromConsole);
		System.out.println("Enroll a finger for badge number " + badgeNumber);
		int firstAskedBadgeNumber = badgeNumber;
		System.out.println();
		byte [] template = {};
		if (devType == BiometricAPI.BIOAPI_COGENT) {		
			template = new byte [BiometricAPI.COGENT_TEMPLATE_SIZE];
		} else if (devType == BiometricAPI.BIOAPI_SUPREMA) {
			template = new byte [BiometricAPI.SUPREMA_TEMPLATE_SIZE];
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			if (subType.i == BiometricAPI.BIOAPI_ONE_TO_ONE) {
				template = new byte [BiometricAPI.BIOSCRYPT_VERIFY_TEMPLATE_SIZE];
			} else {
				template = new byte [BiometricAPI.BIOSCRYPT_IDENTIFY_TEMPLATE_SIZE];
			}
		}
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {
			errCode = bio.BioEnrollTemplate(handle, badgeNumber, (byte)0, bp, template);
			System.out.println(bio.errString(errCode) + " = bio.BioEnrollTemplate(handle, " + badgeNumber + ", (byte)0, bp, template)");
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			errCode = bio.BioEnrollTemplate(handle, badgeNumber, (byte)0, bioscryptQuality, template);
			System.out.println(bio.errString(errCode) + " = bio.BioEnrollTemplate(handle, " + badgeNumber + ", (byte)0, bioscryptQuality, template)");
		}
		if (errCode == BiometricAPI.BIOAPI_OK) {
			for (int i=0; i<2; i++) {
				System.out.print("template[" + Dbg.toHex((char)(i*16)) + ".." + Dbg.toHex((char)(i*16+15)) + "] = ");
				for (int j=0; j<16; j++) {
					System.out.print(Dbg.toHex((char)template[i*16+j]) + " ");
				}
				System.out.println();
			}
			if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {
				System.out.println(bio.errString(bio.BioGetEnrollQuality(handle, ip, bp)) + " = bio.BioGetEnrollQuality(handle, ip, bp)");
				System.out.println("badge " + ip.i + " quality = " + bp.b);
			} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
				System.out.println(bio.errString(bio.BioGetEnrollQuality(handle, ip, bioscryptQuality)) + " = bio.BioGetEnrollQuality(handle, ip, bioscryptQuality)");
				System.out.println("badge " + ip.i + " content = " + bioscryptQuality.content + " quality = " + bioscryptQuality.quality);
			}
			System.out.println();
			System.out.println("remove finger");
			bio.wait(2);
			System.out.println("present badge " + ip.i + " finger for verify");
			System.out.println(bio.errString(bio.BioVerifyTemplate(handle, ip.i, template)) + " = bio.BioVerifyTemplate(handle, " + ip.i + ", template)");
			System.out.println();
		}
		System.out.println(bio.errString(bio.BioGetVerifyScore(handle, ip.i, bp)) + " = bio.BioGetVerifyScore(handle, " + ip.i + ", bp)");
		System.out.println("score = " + bp.b);
		if (errCode == BiometricAPI.BIOAPI_OK) {
			byte [][] templateList = new byte [][] {template};
			System.out.println(bio.errString(bio.BioPutTemplates(handle, templateList)) + "= bio.BioPutTemplates(handle, templateList)");
		}
		System.out.println(bio.errString(bio.BioGetTemplateCount(handle, ip)) + "= bio.BioGetTemplateCount(handle, ip)");
		System.out.println("count = " + ip.i);
		System.out.println(bio.errString(bio.BioGetTemplateLimit(handle, ip)) + "= bio.BioGetTemplateLimit(handle, ip)");
		System.out.println("limit = " + ip.i);
		System.out.println();

		System.out.println("Select a badge number for a secondary enroll:");
		fromConsole = stdin.readLine();
		badgeNumber = Integer.parseInt(fromConsole);
		System.out.println("Enroll a secondary finger for badge number " + badgeNumber);
		System.out.println();
		if (devType == BiometricAPI.BIOAPI_COGENT || devType == BiometricAPI.BIOAPI_SUPREMA) {
			System.out.println(bio.errString(bio.BioEnroll(handle, badgeNumber, (byte)1, bp)) + " = bio.BioEnroll(handle, " + badgeNumber + ", (byte)1, bp)");
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			System.out.println(bio.errString(bio.BioEnroll(handle, badgeNumber, (byte)1, bioscryptQuality)) + " = bio.BioEnroll(handle, " + badgeNumber + ", (byte)1, bioscryptQuality)");
		}
		System.out.println();
		System.out.println("remove finger");
		bio.wait(2);
		System.out.println("present badge " + badgeNumber + " secondary finger for verify");
		System.out.println("(Make this fail if you don't want to delete all templates at the end)");
		errCode = bio.BioVerify(handle, badgeNumber);
		System.out.println(bio.errString(errCode)  + " = bio.BioVerify(handle, " + badgeNumber + ")");
		boolean doDeleteAll = false;
		if (errCode == BiometricAPI.BIOAPI_OK) {
			doDeleteAll = true;
		}
		System.out.println();
		bio.wait(2);
		System.out.println("Getting a list of all templates in the reader");
		System.out.println(bio.errString(bio.BioGetTemplateCount(handle, ip)) + "= bio.BioGetTemplateCount(handle, ip)");
		int count = ip.i;
        int [] badgeNumberList = new int [count];
		byte [] templateIndexList = new byte [count];
		System.out.println(bio.errString(bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)");
		System.out.println(ip.i + " templates found");
		for (int i=0; i<count; i++) {
			System.out.println("(" + badgeNumberList[i] + ", " + templateIndexList[i] + ")");
		}
		System.out.println("Getting a list of all badge numbers with at least one template stored");
		templateIndexList = null;
		System.out.println(bio.errString(bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)");
		System.out.println(ip.i + " distinct badges found");
		for (int i=0; i<ip.i; i++) {
			System.out.print(badgeNumberList[i]);
			if (i != ip.i - 1) {
				System.out.print(", ");
			}
		}
		System.out.println();
		System.out.println("Checking if badges (12345, 0), (" + badgeNumber + ", 1) are in the reader");
		badgeNumberList = new int [] {12345, badgeNumber};
		templateIndexList = new byte [] {0, 1};
		System.out.println(bio.errString(bio.BioAreTemplateIdsPresent(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioAreTemplateIdsPresent(handle, badgeNumberList, templateIndexList, ip)");
		System.out.println("present = " + ip.i);
		System.out.println("Checking if templates for both badges 12345 and " + badgeNumber + " are in the reader");
		badgeNumberList = new int [] {12345, badgeNumber};
		templateIndexList = null;
		System.out.println(bio.errString(bio.BioAreTemplateIdsPresent(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioAreTemplateIdsPresent(handle, badgeNumberList, templateIndexList, ip)");
		System.out.println("present = " + ip.i);
		System.out.println();
		System.out.println("Getting templates (12345,0) and (" + badgeNumber + ", 1)");
		badgeNumberList = new int [] {12345, badgeNumber};
		templateIndexList = new byte [] {0, 1};
		byte [][] templateList = {};
		if (devType == BiometricAPI.BIOAPI_COGENT) {
			templateList = new byte [2][BiometricAPI.COGENT_TEMPLATE_SIZE];
		} else if (devType == BiometricAPI.BIOAPI_SUPREMA) {
			templateList = new byte [2][BiometricAPI.SUPREMA_TEMPLATE_SIZE];
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			if (subType.i == BiometricAPI.BIOAPI_ONE_TO_ONE) {
				templateList = new byte [2][BiometricAPI.BIOSCRYPT_VERIFY_TEMPLATE_SIZE];
			} else {
				templateList = new byte [2][BiometricAPI.BIOSCRYPT_IDENTIFY_TEMPLATE_SIZE];
			}
		}
		errCode = bio.BioGetTemplates(handle, badgeNumberList, templateIndexList, templateList);
		System.out.println(bio.errString(errCode) + "= bio.BioGetTemplates(handle, badgeNumberList, templateIndexList, bioTemplateList)");
		if (errCode == BiometricAPI.BIOAPI_OK) {
			for (int i=0; i<2; i++) {
				if (templateList[i] == null) {
					System.out.println("template (" + badgeNumberList[i] + ", " + templateIndexList[i] + ") not found");
					continue;
				}
				for (int j=0; j<2; j++) {
					System.out.print("template(" + badgeNumberList[i] + ", " + templateIndexList[i] + ")[" + Dbg.toHex((char)(j*16)) + ".." + Dbg.toHex((char)(j*16+15)) + "] = ");
					for (int k=0; k<16; k++) {
						System.out.print(Dbg.toHex((char)templateList[i][j*16+k]) + " ");
					}
					System.out.println();
				}
			}
		}
		System.out.println("Deleting template (12345, 0) and " + badgeNumber + ", 1)");
		System.out.println(bio.errString(bio.BioTemplateDelete(handle, badgeNumberList, templateIndexList)) + "= bio.BioTemplateDelete(handle, badgeNumberList, templateIndexList)");
		badgeNumberList = new int [] {firstAskedBadgeNumber};
		templateIndexList = new byte [] {0};
		if (devType == BiometricAPI.BIOAPI_COGENT) {
			System.out.println("Attempting to set security for template (" + firstAskedBadgeNumber + ", 0) to 50 (expect NOT_SUPPORTED)");
			System.out.println(bio.errString(bio.BioSetTemplateSecurity(handle, badgeNumberList, templateIndexList, (byte)50)) + "= bio.BioSetTemplateSecurity(handle, badgeNumberList, templateIndexList, (byte)50)");
		} else if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
			System.out.println("Attempting to set security for template (" + firstAskedBadgeNumber + ", 0) to 5");
			System.out.println(bio.errString(bio.BioSetTemplateSecurity(handle, badgeNumberList, templateIndexList, (byte)5)) + "= bio.BioSetTemplateSecurity(handle, badgeNumberList, templateIndexList, (byte)5)");
		} else if (devType == BiometricAPI.BIOAPI_SUPREMA) {
			System.out.println("Attempting to set system wide security 0x51");
			System.out.println(bio.errString(bio.BioSetTemplateSecurity(handle, null, null, (byte)0x51)) + "= bio.BioSetTemplateSecurity(handle, null, null, (byte)0x51)");
		}
		System.out.println("Getting all templates for " + firstAskedBadgeNumber);
		errCode = bio.BioGetTemplates(handle, badgeNumberList, null, templateList);
		System.out.println(bio.errString(errCode) + "= bio.BioGetTemplates(handle, badgeNumberList, null, bioTemplateList)");
		if (errCode == BiometricAPI.BIOAPI_OK) {
			for (int i=0; i<2; i++) {
				if (templateList[i] != null) {
					System.out.println("found template (" + firstAskedBadgeNumber + ", " + i + ")");
				}
			}
		}
		badgeNumber = 3;
		badgeNumberList = new int [] {badgeNumber};
		System.out.println("Deleting all templates for " + badgeNumber);
		System.out.println(bio.errString(bio.BioTemplateDelete(handle, badgeNumberList, null)) + "= bio.BioTemplateDelete(handle, badgeNumberList, null)");
        badgeNumberList = new int [2];
		templateIndexList = new byte [2];
		System.out.println("Getting a list of up to 2 templates left in the device");
		System.out.println(bio.errString(bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)");
		System.out.println(ip.i + " templates found");
		for (int i=0; i<ip.i; i++) {
			System.out.println("(" + badgeNumberList[i] + ", " + templateIndexList[i] + ")");
		}
		System.out.println("Calibration attempted only if first entered badge number was 0xCA");
		if (firstAskedBadgeNumber == 0xCA) {
			if (devType == BiometricAPI.BIOAPI_BIOSCRYPT) {
				System.out.println("expected NOT_SUPPORTED");
			}
			System.out.println(bio.errString(bio.BioCalibrate(handle)) + "= bio.BioCalibrate(handle)");
		} else {
			System.out.println("Calibration not attempted");
		}
		// exercise identify mode
		String mq_name = "/BiometricAPIviaJNI.mq";
		errCode = bio.BioSetModeIdentify(handle, mq_name);
		if (devType == BiometricAPI.BIOAPI_BIOSCRYPT &&	subType.i == BiometricAPI.BIOAPI_ONE_TO_ONE) {
			System.out.println("expect NOT_SUPPORTED");
		}
		System.out.println(bio.errString(errCode) + "= bio.BioSetModeIdentify(handle, " + mq_name + ")");
		if (errCode == BiometricAPI.BIOAPI_OK) {
			try {
				MQ mq = new MQ();
				final int identificationModeSeconds = 30;
				System.out.println("All identifications in the next " + identificationModeSeconds +
				                   " seconds will be reported");
				BioVerificationReport_struct idStruct = new BioVerificationReport_struct();
				long startAt = System.currentTimeMillis();
				long now = startAt;
				boolean interruptShown = false;
				boolean pauseCompleted = false;
				int mqdes = mq.mq_open(mq_name, mq.O_RDONLY | mq.O_NONBLOCK);
				if (mqdes == -1) {
					throw new Exception("mq_open failure errno = " + mq.errno);
				}
				MQ.mq_attr_struct mq_attr = mq.new mq_attr_struct();
				if (mq.mq_getattr(mqdes, mq_attr) != 0) {
					throw new Exception("mq_getattr failure errno = " + mq.errno);
				}
				do {
					if (!interruptShown &&
					    now - startAt > identificationModeSeconds * 1000 / 3) {
						System.out.println("Interrupting identify mode to get a count");
						System.out.println(bio.errString(bio.BioGetTemplateCount(handle, ip)) + "= bio.BioGetTemplateCount(handle, ip)");
						System.out.println(ip.i + " templates left in the device");
						interruptShown = true;
					}
					if (!pauseCompleted &&
						now - startAt > 2 * identificationModeSeconds * 1000 / 3) {
						System.out.println("Pausing identification mode for 5 seconds");
						System.out.println(bio.errString(bio.BioSetModeVerify(handle)) + "= bio.BioSetVerifyMode(handle)");
						BioDiagnostics_struct diag = new BioDiagnostics_struct();
						System.out.println(bio.errString(bio.BioDiagnostics(handle, diag)) + "= bio.bio.BioDiagnostics(handle, diag)");
						System.out.println("diag.badEnrolls=" + diag.badEnrolls + " .validEnrolls=" + diag.validEnrolls);
						System.out.println("    .badReads=" + diag.badReads + " .validReads=" + diag.validReads);
						System.out.println("    .powerCycleCount=" + diag.powerCycleCount + " .powerCycleTime=" + diag.powerCycleTime);
						System.out.println("Getting a list of all templates in the reader");
						System.out.println(bio.errString(bio.BioGetTemplateCount(handle, ip)) + "= bio.BioGetTemplateCount(handle, ip)");
						count = ip.i;
				        badgeNumberList = new int [count];
						templateIndexList = new byte [count];
						System.out.println(bio.errString(bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)) + "= bio.BioGetAllTemplateIds(handle, badgeNumberList, templateIndexList, ip)");
						System.out.println(ip.i + " templates found");
						for (int i=0; i<count; i++) {
							System.out.println("(" + badgeNumberList[i] + ", " + templateIndexList[i] + ")");
						}
						bio.wait(5);
						System.out.println("Resuming identification mode");
						System.out.println(bio.errString(bio.BioSetModeIdentify(handle, mq_name)) + "= bio.BioSetModeIdentify(handle, " + mq_name + ")");
						pauseCompleted = true;
					}
					byte [] buffer = new byte [(int)mq_attr.mq_msgsize];
					mq.errno = 0;
					int msgSize = mq.mq_receive(mqdes, buffer);
					if (mq.errno != 0 && mq.errno != mq.EAGAIN) {
						throw new Exception("mq_receive failure errno = " + mq.errno);
					}
					if (mq.errno == mq.EAGAIN) {
						bio.waitMilli(100); // wait a 10th of a second
					} else {
						if (msgSize < 3) {
							throw new Exception("invalid message (len = " + msgSize + ")");
						}
						if (buffer[msgSize-1] != 0) {
							throw new Exception("badge number string in message not terminated");
						}
						idStruct.score = buffer[0];
						idStruct.index = buffer[1];
 						int nullFoundAt = -1;
						for (int i=0; i<msgSize-2; i++) {
							idStruct.badge[i] = buffer[2+i];
							if (buffer[2+i] == 0) {
								nullFoundAt = i;
								break;
							}
						}
						System.out.println();
						String badgeNumberString = new String(idStruct.badge, 0, nullFoundAt, "US-ASCII");
						System.out.println("badgeNumberString = " + badgeNumberString + " len = " + badgeNumberString.length());
						if (badgeNumberString.equals("")) {
							System.out.print("Saw failed identification attempt (code " + idStruct.index + ")");
						} else {
							badgeNumber = Integer.parseInt(badgeNumberString);
							System.out.print("saw badge " + badgeNumber + " index " + idStruct.index + " score " + idStruct.score);
						}
						System.out.println(" after " + (now-startAt)/1000 + "." + (now-startAt)%1000 + " seconds");
					}
					now = System.currentTimeMillis();
				} while (now-startAt < identificationModeSeconds*1000);
				System.out.println("Exiting identification mode");
				System.out.println(bio.errString(bio.BioSetModeIdentify(handle, null)) + "= bio.BioSetModeIdentify(handle, null)");
				mq.mq_unlink(mq_name);
			} catch (Exception ex) {
				System.out.println("Exception " + ex + " encountered trying to use Identify mode");
				ex.printStackTrace();
			}
		}
		if (doDeleteAll) {
			System.out.println("Deleting all templates");
			System.out.println(bio.errString(bio.BioTemplateDelete(handle, null, null)) + "= bio.BioTemplateDelete(handle, null, null)");
		}
		System.out.println(bio.errString(bio.BioGetTemplateCount(handle, ip)) + "= bio.BioGetTemplateCount(handle, ip)");
		System.out.println(ip.i + " templates left in the device");
	}
}
