/*
+====================================================================+
|Global Terminal                                                             |
|  Written by:  Mike Scudder/MicroTools                               |
+====================================================================+
|                                                                    |
|  2007 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/

// Documentation:
//
// The Biometric API is documented in BiometericAPI.h.
//
// That documentation is in reference to the C language.
//
// This Java interface follows closely to the C calls, with the following
// exceptions:
//
// 1. Pointers to C languare uint32_t are replaced with the class IntPass,
//    since Java is exclusively pass by value.  In cases where a single
//    uint8_t value is passed back, the class BytePass is used.
//
// 2. Pointers to C languare strings, or char *, are replaced with the class
//    StringPass, since Java is exclusively pass by value.
//
// 3. Badge numbers are passed as Java ints, rather than strings
//    which decode into unsigned 32 bit values.
//
// 4. Where the C language interface had a single call with a void * parameter
//    used by both the Bioscrypt and the Cogent biometric devices, the Java
//    interface uses method overloading.  Where a parameter will always be null
//    for the Biometric devices, it is not included. 
//
// 5. Where the C language interface requires a max parameter when passing back
//    an array of values, the Java interface uses the size of the array passed
//    as the maximum number of values that can be passed back.  A count parameter
//    has been added to pass back the number of values passed back, in the case
//    where the C interface used a null pointer to indicate the end of a list.
//
// 6. The Java Native Interface layer creates temporary buffers before calling
//    some C language routines.  Therefore the BIOAPI_OUT_OF_MEMORY error
//    condition may be returned even when not specified in the BiometricAPI.h
//    documentation.
//
// 7. Where the C language interface expects the programmer to allocate correct
//    size buffers for holding templates, the Java interface accepts any size
//    Java byte arrays and ignores extra bytes and discards bytes that do not
//    fit.  Internally it uses temporaries corresponding to maximu size
//    templates, that is, Bioscrypt identify style templates.
//
package com.accu_time.util;

public interface BiometricAPI {
	int BIOAPI_OK                       = (0);
	int BIOAPI_BUSY                     = (BIOAPI_OK - 1);
	int BIOAPI_INVALID                  = (BIOAPI_BUSY - 1);
	int BIOAPI_OUT_OF_HANDLES           = (BIOAPI_INVALID - 1);
	int BIOAPI_IN_USE                   = (BIOAPI_OUT_OF_HANDLES - 1);
	int BIOAPI_NOT_SUPPORTED            = (BIOAPI_IN_USE - 1);
	int BIOAPI_NO_COMMUNICATIONS        = (BIOAPI_NOT_SUPPORTED - 1);
	int BIOAPI_SELF_TEST_FAILED         = (BIOAPI_NO_COMMUNICATIONS - 1);
	int BIOAPI_NOT_PRESENTED            = (BIOAPI_SELF_TEST_FAILED - 1);
	int BIOAPI_ENROLL_FAILED            = (BIOAPI_NOT_PRESENTED - 1);
	int BIOAPI_VERIFY_FAILED            = (BIOAPI_ENROLL_FAILED - 1);
	int BIOAPI_NO_PREVIOUS_ENROLL       = (BIOAPI_VERIFY_FAILED - 1);
	int BIOAPI_NO_PREVIOUS_VERIFY       = (BIOAPI_NO_PREVIOUS_ENROLL - 1);
	int BIOAPI_FLASH_ERROR_VERSION      = (BIOAPI_NO_PREVIOUS_VERIFY - 1);
	int BIOAPI_FLASH_ERROR_TIMEOUT      = (BIOAPI_FLASH_ERROR_VERSION - 1);
	int BIOAPI_FLASH_ERROR_BAD_REC      = (BIOAPI_FLASH_ERROR_TIMEOUT - 1);
	int BIOAPI_FLASH_LOAD_REP_REC       = (BIOAPI_FLASH_ERROR_BAD_REC - 1);
	int BIOAPI_FLASH_LOAD_FAILED        = (BIOAPI_FLASH_LOAD_REP_REC - 1);
	int BIOAPI_FLASH_LOAD_BAD_SEQ       = (BIOAPI_FLASH_LOAD_FAILED - 1);
	int BIOAPI_FLASH_ERROR_RDR_BAD_COMM = (BIOAPI_FLASH_LOAD_BAD_SEQ - 1);
	int BIOAPI_MESSAGE_QUEUE_PROBLEM    = (BIOAPI_FLASH_ERROR_RDR_BAD_COMM - 1);
	int BIOAPI_OUT_OF_MEMORY            = (BIOAPI_MESSAGE_QUEUE_PROBLEM - 1);
	int BIOAPI_TEMPLATE_STORAGE_PROBLEM = (BIOAPI_OUT_OF_MEMORY - 1);

	String BioAPI_errMsgs[] = {
		"OK",
		"BUSY",
		"INVALID",
		"OUT_OF_HANDLES",
		"IN_USE",
		"NOT_SUPPORTED",
		"NO_COMMUNICATIONS",
		"SELF_TEST_FAILED",
		"NOT_PRESENTED",
		"ENROLL_FAILED",
		"VERIFY_FAILED",
		"NO_PREVIOUS_ENROLL",
		"NO_PREVIOUS_VERIFY",
		"FLASH_ERROR_VERSION",
		"FLASH_ERROR_TIMEOUT",
		"FLASH_ERROR_BAD_REC",
		"FLASH_LOAD_REP_REC",
		"FLASH_LOAD_FAILED",
		"FLASH_LOAD_BAD_SEQ",
		"FLASH_ERROR_RDR_BAD_COMM ",
		"MESSAGE_QUEUE_PROBLEM",
		"OUT_OF_MEMORY",
		"TEMPLATE_STORAGE_PROBLEM ",
	};

	int BIOAPI_COGENT = 0;
	int BIOAPI_BIOSCRYPT = 1;
	int BIOAPI_SUPREMA = 2;
	int BIOAPI_LUMIDIGM = 3;

	int BIOAPI_ONE_TO_ONE = 0;
	int BIOAPI_ONE_TO_MANY = 1;

/*
#define FPRNT_INPHASE_LEN_IDENTIFY 2288
#define FPRNT_INPHASE_LEN_VERIFY   284	 

typedef struct FPRNT_TEMPLATE_HEADER
{
   uchar sensor_version;
   uchar template_version;       // 2 for 1:1, 3 for 1:many (identify)
   char  name[16];
   uchar finger;
   uchar admin_level;
   uchar schedule;
   uchar security_thersh;
   uchar noise_level[18];
   uchar corramb[3];
   uchar ihcore[3];
   uchar ivcore[3];
   uchar temp_xoffset;
   uchar temp_yoffset;
   uchar index;
} FPRNT_TEMPLATE_HEADER;

typedef struct FPRNT_TEMPLATE_VERIFY
{
   ulong id;
   ulong employee_id;
   ulong password;
   FPRNT_TEMPLATE_HEADER H;
   uchar inphase[FPRNT_INPHASE_LEN_VERIFY];       // 1:1
} FPRNT_TEMPLATE_VERIFY;

typedef struct FPRNT_TEMPLATE_BIOSCRYPT
{
   ulong id;
   ulong employee_id;
   ulong password;
   FPRNT_TEMPLATE_HEADER H;
   uchar inphase[FPRNT_INPHASE_LEN_IDENTIFY];     // 1:many
} FPRNT_TEMPLATE_BIOSCRYPT;

#define MINUTIAE_TEMPLATE_SIZE 752           // compressed minutiae templates
#define ID_LEN_COGENT           10

#define COGENT_BADGE_ID_OFFSET     0	// Offset into template header
#define COGENT_BADGE_ID_LENGTH    10
#define COGENT_BADGE_INDEX_OFFSET 11    // Offset into template header
#define COGENT_BADGE_INDEX_LENGTH  1

typedef struct FPRNT_HEADER_COGENT
{
     char  id_str[ID_LEN_COGENT];     // 10 byte id str
     char  index[2];                  // _0, _1, ....  
     uchar  ctrl_code[10];
     uchar  app_data[9];
     uchar reserved;
} FPRNT_HEADER_COGENT;

typedef struct FPRNT_TEMPLATE_COGENT
{
     FPRNT_HEADER_COGENT header;    // 32 byte id str
     uchar  minutiae[MINUTIAE_TEMPLATE_SIZE];
} FPRNT_TEMPLATE_COGENT;

#define SU_MINUTIAE_TEMPLATE_SIZE 384           // minutiae templates

typedef struct FPRNT_HEADER_SUPREMA
     {
     ulong  id;
     uchar  index; 
     uchar  security_level;
     uchar  spare1;
     uchar  spare2;   
     } __attribute__((__packed__)) FPRNT_HEADER_SUPREMA;

typedef enum {
	SUPREMA_FAR_1_IN_10 = 0x30,  // used internally as 1 in 10000
	SUPREMA_FAR_1_IN_100,        // used internally as 1 in 10000
	SUPREMA_FAR_1_IN_10000,      // used internally as 1 in 10000
	SUPREMA_FAR_1_IN_100000,
	SUPREMA_FAR_1_IN_1000000,
	SUPREMA_FAR_1_IN_10000000,
	SUPREMA_FAR_1_IN_100000000,
	SUPREMA_FAR_3_IN_100 = 0x40, // used internally as 1 in 10000
	SUPREMA_FAR_3_IN_1000,		 // used internally as 1 in 10000
	SUPREMA_FAR_3_IN_10000,		 // used internally as 1 in 10000
	SUPREMA_FAR_3_IN_100000,
	SUPREMA_FAR_3_IN_1000000,
	SUPREMA_FAR_3_IN_10000000,
	SUPREMA_FAR_3_IN_100000000,
	SUPREMA_AUTOMATIC_NORMAL = 0x50,
	SUPREMA_AUTOMATIC_SECURE,
	SUPREMA_AUTOMATIC_MORE_SECURE,
	} SupremaSecurity;

typedef struct FPRNT_TEMPLATE_SUPREMA
     {
     FPRNT_HEADER_SUPREMA   header;   
     uchar                  minutiae[SU_MINUTIAE_TEMPLATE_SIZE];
     } __attribute__((__packed__)) FPRNT_TEMPLATE_SUPREMA;

#define FPRNT_TEMPLATE_XFER_PACKET_LEN (sizeof(FPRNT_TEMPLATE_HEADER)+12)

//#define FPRNT_MAX_TEMPLATE_LIST (sizeof(ulong) * 4000) + 32            // sizeof .id

#define FPRNT_BS_UNIT_VERIFY       0    // fprnt_unit_type = 1:1
#define FPRNT_BS_UNIT_IDENTIFY     1    // fprnt_unit_type = 1:many

#define FPRNT_TMPLT_VER_VERIFY     2    // fprnt template version for verify
#define FPRNT_TMPLT_VER_IDENTIFY   3    // fprnt template version for identify

typedef union FPRNT_TEMPLATE
{
     FPRNT_TEMPLATE_BIOSCRYPT bs;
     FPRNT_TEMPLATE_COGENT    cs;
     FPRNT_TEMPLATE_SUPREMA   su;
} FPRNT_TEMPLATE;
*/
	int COGENT_TEMPLATE_SIZE = 10 + 2 + 10 + 9 + 1 + 752;
    
	int BIOSCRYPT_NUM_THRESHOLDS = (3);
	int BIOSCRYPT_TEMPLATE_HEADER_SIZE = 2*1 + 16 + 4*1 + 18 + 3*3 + 3*1;
	int BIOSCRYPT_VERIFY_TEMPLATE_SIZE = 3*4 + BIOSCRYPT_TEMPLATE_HEADER_SIZE + 284;
	int BIOSCRYPT_IDENTIFY_TEMPLATE_SIZE = 3*4 + BIOSCRYPT_TEMPLATE_HEADER_SIZE + 2288;

	int SUPREMA_TEMPLATE_SIZE = 4 + 1 + 1 + 1 + 1 + 384;

	int LUMIDIGM_TEMPLATE_SIZE = 3*512 + 512; //3 x 512 plus minutae, plus header info

	class BioDiagnostics_struct
	{
		public int validReads;
		public int validEnrolls;
		public int badReads;
		public int badEnrolls;
		public int powerCycleCount;
		public long powerCycleTime;
	}

	class BioscryptQuality_struct {
		public byte quality;
		public byte content;
		public byte [] qualityThresholds;
		public byte [] contentThresholds;
	}
		
	class BioVerificationReport_struct {
		public byte score;
		public byte index;
		public byte badge [];
		public BioVerificationReport_struct() {
			score = 0;
			index = 0;
			badge = new byte [1000];
			for (int i=0; i<1000; i++) {
				badge[i] = 0;
			}
		}
	}

	class StringPass {
		public String s;
		public StringPass() {
			this.s = null;
		}
		public StringPass(String s) {
			this.s = s;
		}
	}

	class IntPass {
		public int i;
		public IntPass() {
			this.i = 0;
		}
		public IntPass(int i) {
			this.i = i;
		}
	}

	class BytePass {
		public byte b;
		public BytePass() {
			this.b = 0;
		}
		public BytePass(byte b) {
			this.b = b;
		}
	}
	void wait(int seconds);
	void waitMilli(int milliseconds);
	
	int BioInitialization(IntPass handle, int deviceType, byte comPort);
	int BioInitComPort(int handle, int baudRate);
	int BioPowerOn(int handle);
	int BioPowerOff(int handle);
	int BioDiagnostics(int handle, BioDiagnostics_struct diagnostics);
	int BioSelfTest(int handle, StringPass selfTestResult);
	int BioDeviceResponsive(int handle);
	int BioGetVersionString(int handle, StringPass versionString);
	int BioGetDeviceSubType(int handle, IntPass deviceSubType);
	int BioFlashLoad(int handle, byte [] programRecord, IntPass recordNumber, IntPass done);
	int BioEnroll(int handle, int badgeNumber, byte templateIndex, BioscryptQuality_struct quality);
	int BioEnroll(int handle, int badgeNumber, byte templateIndex, BytePass quality);  // for Cogent
	int BioGetEnrollQuality(int handle, IntPass badgeNumber, BioscryptQuality_struct quality);
	int BioGetEnrollQuality(int handle, IntPass badgeNumber, BytePass quality);  // for Cogent
	int BioEnrollTemplate(int handle, int badgeNumber, byte templateIndex, BioscryptQuality_struct quality, byte [] template);
	int BioEnrollTemplate(int handle, int badgeNumber, byte templateIndex, BytePass quality, byte [] template);  // for Cogent
	int BioVerify(int handle, int badgeNumber);
	int BioVerifyTemplate(int handle, int badgeNumber, byte [] template);
	int BioVerifyAgainstTemplates(int handle, byte [][] bioTemplateList);
	int BioGetVerifyScore(int handle, int badgeNumber, BytePass score);
	int BioGetTemplates(int handle, int [] badgeNumberList, byte [] templateIndexList, byte [][]bioTemplateList);
	int BioPutTemplates(int handle, byte [][] bioTemplateList);
	int BioGetTemplateCount(int handle, IntPass count);
	int BioGetTemplateLimit(int handle, IntPass limit);
	int BioGetAllTemplateIds(int handle, int [] badgeNumberList, byte [] templateIndexList, IntPass count);
	int BioAreTemplateIdsPresent(int handle, int [] badgeNumberList, byte [] templateIndexList, IntPass present);
	int BioTemplateDelete(int handle, int [] badgeNumberList, byte [] templateIndexList);
	int BioSetTemplateSecurity(int handle, int [] badgeNumberList, byte [] templateIndexList, byte verificationThreshold);
	int BioSetModeIdentify(int handle, String bioMsgQueueName);
	int BioSetModeVerify(int handle);
	int BioCalibrate(int handle);	 // Only Cogents can be calibrated
	int BioThreshold(int handle, int threshold);
	int BioGetThreshold(int handle, IntPass threshold);
}
