/*
+====================================================================+
|Global Terminal                                                             |
|  Written by:  Mike Scudder/MicroTools                               |
+====================================================================+
|                                                                    |
|  2007 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/
package com.accu_time.util;

import java.io.FileOutputStream;
import java.text.DecimalFormat;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.util.StringTokenizer;

public class Keypad {
	final public int K_RAW       = 0x00;
	final public int K_XLATE     = 0x01;
	final public int K_MEDIUMRAW = 0x02;
	final public int K_UNICODE   = 0x03;
	final public int KEYPAD_ROWS = 6;
	final public int KEYPAD_COLUMNS = 9;

	private class KeyCodeChar {
		String keyName;
		char unicodeChar;
		int keypadValue;

		KeyCodeChar(String keyName, char unicodeChar, int keypadValue) {
			this.keyName = keyName;
			this.unicodeChar = unicodeChar;
			this.keypadValue = keypadValue;
		}			
	}

    static public KeyCodeChar keyTriples[];
    {
    keyTriples = new KeyCodeChar [] {
	  new KeyCodeChar("KEY_RESERVED", '\0', 0),
	  new KeyCodeChar("KEY_ESC", (char)0x1b, 1),
	  new KeyCodeChar("KEY_1", '1', 2),
	  new KeyCodeChar("KEY_2", '2', 3),
	  new KeyCodeChar("KEY_3", '3', 4),
	  new KeyCodeChar("KEY_4", '4', 5),
	  new KeyCodeChar("KEY_5", '5', 6),
	  new KeyCodeChar("KEY_6", '6', 7),
	  new KeyCodeChar("KEY_7", '7', 8),
	  new KeyCodeChar("KEY_8", '8', 9),
	  new KeyCodeChar("KEY_9", '9', 10),
	  new KeyCodeChar("KEY_0", '0', 11),
	  new KeyCodeChar("KEY_MINUS", '-', 12),
	  new KeyCodeChar("KEY_EQUAL", '=', 13),
	  new KeyCodeChar("KEY_BACKSPACE", '\b', 14),
	  new KeyCodeChar("KEY_TAB", '\t', 15),
	  new KeyCodeChar("KEY_Q", 'Q', 16),
	  new KeyCodeChar("KEY_W", 'W', 17),
	  new KeyCodeChar("KEY_E", 'E', 18),
	  new KeyCodeChar("KEY_R", 'R', 19),
	  new KeyCodeChar("KEY_T", 'T', 20),
	  new KeyCodeChar("KEY_Y", 'Y', 21),
	  new KeyCodeChar("KEY_U", 'U', 22),
	  new KeyCodeChar("KEY_I", 'I', 23),
	  new KeyCodeChar("KEY_O", 'O', 24),
	  new KeyCodeChar("KEY_P", 'P', 25),
	  new KeyCodeChar("KEY_LEFTBRACE", '{', 26),
	  new KeyCodeChar("KEY_RIGHTBRACE", '}', 27),
	  new KeyCodeChar("KEY_ENTER", '\r', 28),
	  new KeyCodeChar("KEY_LEFTCTRL", '\0', 29),
	  new KeyCodeChar("KEY_A", 'A', 30),
	  new KeyCodeChar("KEY_S", 'S', 31),
	  new KeyCodeChar("KEY_D", 'D', 32),
	  new KeyCodeChar("KEY_F", 'F', 33),
	  new KeyCodeChar("KEY_G", 'G', 34),
	  new KeyCodeChar("KEY_H", 'H', 35),
	  new KeyCodeChar("KEY_J", 'J', 36),
	  new KeyCodeChar("KEY_K", 'K', 37),
	  new KeyCodeChar("KEY_L", 'L', 38),
	  new KeyCodeChar("KEY_SEMICOLON", ';', 39),
	  new KeyCodeChar("KEY_APOSTROPHE", '\'', 40),
	  new KeyCodeChar("KEY_GRAVE", '`', 41),
	  new KeyCodeChar("KEY_LEFTSHIFT", '\0', 42),
	  new KeyCodeChar("KEY_BACKSLASH", '\\', 43),
	  new KeyCodeChar("KEY_Z", 'Z', 44),
	  new KeyCodeChar("KEY_X", 'X', 45),
	  new KeyCodeChar("KEY_C", 'C', 46),
	  new KeyCodeChar("KEY_V", 'V', 47),
	  new KeyCodeChar("KEY_B", 'B', 48),
	  new KeyCodeChar("KEY_N", 'N', 49),
	  new KeyCodeChar("KEY_M", 'M', 50),
	  new KeyCodeChar("KEY_COMMA", ',', 51),
	  new KeyCodeChar("KEY_DOT", '.', 52),
	  new KeyCodeChar("KEY_SLASH", '/', 53),
	  new KeyCodeChar("KEY_RIGHTSHIFT", '\0', 54),
	  new KeyCodeChar("KEY_KPASTERISK", '*', 55),
	  new KeyCodeChar("KEY_LEFTALT", '\0', 56),
	  new KeyCodeChar("KEY_SPACE", ' ', 57),
	  new KeyCodeChar("KEY_CAPSLOCK", '\0', 58),
	  new KeyCodeChar("KEY_F1", '\0', 59),
	  new KeyCodeChar("KEY_F2", '\0', 60),
	  new KeyCodeChar("KEY_F3", '\0', 61),
	  new KeyCodeChar("KEY_F4", '\0', 62),
	  new KeyCodeChar("KEY_F5", '\0', 63),
	  new KeyCodeChar("KEY_F6", '\0', 64),
	  new KeyCodeChar("KEY_F7", '\0', 65),
	  new KeyCodeChar("KEY_F8", '\0', 66),
	  new KeyCodeChar("KEY_F9", '\0', 67),
	  new KeyCodeChar("KEY_F10", '\0', 68),
	  new KeyCodeChar("KEY_NUMLOCK", '\0', 69),
	  new KeyCodeChar("KEY_SCROLLLOCK", '\0', 70),
	  new KeyCodeChar("KEY_KP7", '7', 71),
	  new KeyCodeChar("KEY_KP8", '8', 72),
	  new KeyCodeChar("KEY_KP9", '9', 73),
	  new KeyCodeChar("KEY_KPMINUS", '-', 74),
	  new KeyCodeChar("KEY_KP4", '\0', 75),
	  new KeyCodeChar("KEY_KP5", '\0', 76),
	  new KeyCodeChar("KEY_KP6", '\0', 77),
	  new KeyCodeChar("KEY_KPPLUS", '+', 78),
	  new KeyCodeChar("KEY_KP1", '\0', 79),
	  new KeyCodeChar("KEY_KP2", '\0', 80),
	  new KeyCodeChar("KEY_KP3", '\0', 81),
	  new KeyCodeChar("KEY_KP0", '\0', 82),
	  new KeyCodeChar("KEY_KPDOT", '.', 83),
	  new KeyCodeChar("KEY_ZENKAKUHANKAKU", '\0', 85),
	  new KeyCodeChar("KEY_102ND", '\0', 86),
	  new KeyCodeChar("KEY_F11", '\0', 87),
	  new KeyCodeChar("KEY_F12", '\0', 88),
	  new KeyCodeChar("KEY_RO", '\0', 89),
	  new KeyCodeChar("KEY_KATAKANA", '\0', 90),
	  new KeyCodeChar("KEY_HIRAGANA", '\0', 91),
	  new KeyCodeChar("KEY_HENKAN", '\0', 92),
	  new KeyCodeChar("KEY_KATAKANAHIRAGANA", '\0', 93),
	  new KeyCodeChar("KEY_MUHENKAN", '\0', 94),
	  new KeyCodeChar("KEY_KPJPCOMMA", ',', 95),
	  new KeyCodeChar("KEY_KPENTER", '\r', 96),
	  new KeyCodeChar("KEY_RIGHTCTRL", '\0', 97),
	  new KeyCodeChar("KEY_KPSLASH", '/', 98),
	  new KeyCodeChar("KEY_SYSRQ", '\0', 99),
	  new KeyCodeChar("KEY_RIGHTALT", '\0', 100),
	  new KeyCodeChar("KEY_LINEFEED", '\n', 101),
	  new KeyCodeChar("KEY_HOME", '\0', 102),
	  new KeyCodeChar("KEY_UP", '\0', 103),
	  new KeyCodeChar("KEY_PAGEUP", '\0', 104),
	  new KeyCodeChar("KEY_LEFT", '\0', 105),
	  new KeyCodeChar("KEY_RIGHT", '\0', 106),
	  new KeyCodeChar("KEY_END", '\0', 107),
	  new KeyCodeChar("KEY_DOWN", '\0', 108),
	  new KeyCodeChar("KEY_PAGEDOWN", '\0', 109),
	  new KeyCodeChar("KEY_INSERT", '\0', 110),
	  new KeyCodeChar("KEY_DELETE", (char)0x7f, 111),
	  new KeyCodeChar("KEY_MACRO", '\0', 112),
	  new KeyCodeChar("KEY_MUTE", '\0', 113),
	  new KeyCodeChar("KEY_VOLUMEDOWN", '\0', 114),
	  new KeyCodeChar("KEY_VOLUMEUP", '\0', 115),
	  new KeyCodeChar("KEY_POWER", '\0', 116),
	  new KeyCodeChar("KEY_KPEQUAL", '=', 117),
	  new KeyCodeChar("KEY_KPPLUSMINUS", '\0', 118),
	  new KeyCodeChar("KEY_PAUSE", '\0', 119),
	  new KeyCodeChar("KEY_KPCOMMA", ',', 121),
	  new KeyCodeChar("KEY_HANGUEL", '\0', 122),
	  new KeyCodeChar("KEY_HANJA", '\0', 123),
	  new KeyCodeChar("KEY_YEN", (char)0xa5, 124),
	  new KeyCodeChar("KEY_LEFTMETA", '\0', 125),
	  new KeyCodeChar("KEY_RIGHTMETA", '\0', 126),
	  new KeyCodeChar("KEY_COMPOSE", '\0', 127),
	  new KeyCodeChar("KEY_STOP", '\0', 128),
	  new KeyCodeChar("KEY_AGAIN", '\0', 129),
	  new KeyCodeChar("KEY_PROPS", '\0', 130),
	  new KeyCodeChar("KEY_UNDO", '\0', 131),
	  new KeyCodeChar("KEY_FRONT", '\0', 132),
	  new KeyCodeChar("KEY_COPY", '\0', 133),
	  new KeyCodeChar("KEY_OPEN", '\0', 134),
	  new KeyCodeChar("KEY_PASTE", '\0', 135),
	  new KeyCodeChar("KEY_FIND", '\0', 136),
	  new KeyCodeChar("KEY_CUT", '\0', 137),
	  new KeyCodeChar("KEY_HELP", '\0', 138),
	  new KeyCodeChar("KEY_MENU", '\0', 139),
	  new KeyCodeChar("KEY_CALC", '\0', 140),
	  new KeyCodeChar("KEY_SETUP", '\0', 141),
	  new KeyCodeChar("KEY_SLEEP", '\0', 142),
	  new KeyCodeChar("KEY_WAKEUP", '\0', 143),
	  new KeyCodeChar("KEY_FILE", '\0', 144),
	  new KeyCodeChar("KEY_SENDFILE", '\0', 145),
	  new KeyCodeChar("KEY_DELETEFILE", '\0', 146),
	  new KeyCodeChar("KEY_XFER", '\0', 147),
	  new KeyCodeChar("KEY_PROG1", '\0', 148),
	  new KeyCodeChar("KEY_PROG2", '\0', 149),
	  new KeyCodeChar("KEY_WWW", '\0', 150),
	  new KeyCodeChar("KEY_MSDOS", '\0', 151),
	  new KeyCodeChar("KEY_COFFEE", '\0', 152),
	  new KeyCodeChar("KEY_DIRECTION", '\0', 153),
	  new KeyCodeChar("KEY_CYCLEWINDOWS", '\0', 154),
	  new KeyCodeChar("KEY_MAIL", '\0', 155),
	  new KeyCodeChar("KEY_BOOKMARKS", '\0', 156),
	  new KeyCodeChar("KEY_COMPUTER", '\0', 157),
	  new KeyCodeChar("KEY_BACK", '\0', 158),
	  new KeyCodeChar("KEY_FORWARD", '\0', 159),
	  new KeyCodeChar("KEY_CLOSECD", '\0', 160),
	  new KeyCodeChar("KEY_EJECTCD", '\0', 161),
	  new KeyCodeChar("KEY_EJECTCLOSECD", '\0', 162),
	  new KeyCodeChar("KEY_NEXTSONG", '\0', 163),
	  new KeyCodeChar("KEY_PLAYPAUSE", '\0', 164),
	  new KeyCodeChar("KEY_PREVIOUSSONG", '\0', 165),
	  new KeyCodeChar("KEY_STOPCD", '\0', 166),
	  new KeyCodeChar("KEY_RECORD", '\0', 167),
	  new KeyCodeChar("KEY_REWIND", '\0', 168),
	  new KeyCodeChar("KEY_PHONE", '\0', 169),
	  new KeyCodeChar("KEY_ISO", '\0', 170),
	  new KeyCodeChar("KEY_CONFIG", '\0', 171),
	  new KeyCodeChar("KEY_HOMEPAGE", '\0', 172),
	  new KeyCodeChar("KEY_REFRESH", '\0', 173),
	  new KeyCodeChar("KEY_EXIT", '\0', 174),
	  new KeyCodeChar("KEY_MOVE", '\0', 175),
	  new KeyCodeChar("KEY_EDIT", '\0', 176),
	  new KeyCodeChar("KEY_SCROLLUP", '\0', 177),
	  new KeyCodeChar("KEY_SCROLLDOWN", '\0', 178),
	  new KeyCodeChar("KEY_KPLEFTPAREN", '(', 179),
	  new KeyCodeChar("KEY_KPRIGHTPAREN", ')', 180),
	  new KeyCodeChar("KEY_F13", '\0', 183),
	  new KeyCodeChar("KEY_F14", '\0', 184),
	  new KeyCodeChar("KEY_F15", '\0', 185),
	  new KeyCodeChar("KEY_F16", '\0', 186),
	  new KeyCodeChar("KEY_F17", '\0', 187),
	  new KeyCodeChar("KEY_F18", '\0', 188),
	  new KeyCodeChar("KEY_F19", '\0', 189),
	  new KeyCodeChar("KEY_F20", '\0', 190),
	  new KeyCodeChar("KEY_F21", '\0', 191),
	  new KeyCodeChar("KEY_F22", '\0', 192),
	  new KeyCodeChar("KEY_F23", '\0', 193),
	  new KeyCodeChar("KEY_F24", '\0', 194),
	  new KeyCodeChar("KEY_PLAYCD", '\0', 200),
	  new KeyCodeChar("KEY_PAUSECD", '\0', 201),
	  new KeyCodeChar("KEY_PROG3", '\0', 202),
	  new KeyCodeChar("KEY_PROG4", '\0', 203),
	  new KeyCodeChar("KEY_SUSPEND", '\0', 205),
	  new KeyCodeChar("KEY_CLOSE", '\0', 206),
	  new KeyCodeChar("KEY_PLAY", '\0', 207),
	  new KeyCodeChar("KEY_FASTFORWARD", '\0', 208),
	  new KeyCodeChar("KEY_BASSBOOST", '\0', 209),
	  new KeyCodeChar("KEY_PRINT", '\0', 210),
	  new KeyCodeChar("KEY_HP", '\0', 211),
	  new KeyCodeChar("KEY_CAMERA", '\0', 212),
	  new KeyCodeChar("KEY_SOUND", '\0', 213),
	  new KeyCodeChar("KEY_QUESTION", '\0', 214),
	  new KeyCodeChar("KEY_EMAIL", '\0', 215),
	  new KeyCodeChar("KEY_CHAT", '\0', 216),
	  new KeyCodeChar("KEY_SEARCH", '\0', 217),
	  new KeyCodeChar("KEY_CONNECT", '\0', 218),
	  new KeyCodeChar("KEY_FINANCE", '\0', 219),
	  new KeyCodeChar("KEY_SPORT", '\0', 220),
	  new KeyCodeChar("KEY_SHOP", '\0', 221),
	  new KeyCodeChar("KEY_ALTERASE", '\0', 222),
	  new KeyCodeChar("KEY_CANCEL", '\0', 223),
	  new KeyCodeChar("KEY_BRIGHTNESSDOWN", '\0', 224),
	  new KeyCodeChar("KEY_BRIGHTNESSUP", '\0', 225),
	  new KeyCodeChar("KEY_MEDIA", '\0', 226),
	  new KeyCodeChar("KEY_SWITCHVIDEOMODE", '\0', 227),
	  new KeyCodeChar("KEY_KBDILLUMTOGGLE", '\0', 228),
	  new KeyCodeChar("KEY_KBDILLUMDOWN", '\0', 229),
	  new KeyCodeChar("KEY_KBDILLUMUP", '\0', 230),
	  new KeyCodeChar("KEY_UNKNOWN", '\0', 240),
	};
	}

	static {
	  System.loadLibrary("ATSJavaAPI");
	}

	public Keypad() {
	}
	public native void setKeypadRepeatRate(int delay, int period);	  // each in milliseconds
	public native int  ioctl_KDGKBMODE(int[] keypadMode);            // returns keypad mode in cell [0]
	public native int  ioctl_KEYPAD_GMAP(char [][]keyPadDef);         // gets keypad map in [6][9]
	public native int  ioctl_KDSKBMODE(int keypadMode);

	public char [][] getKeypadMap() {
		char [][] map = new char[KEYPAD_ROWS][KEYPAD_COLUMNS];
		int ret = ioctl_KEYPAD_GMAP(map);
		if (ret == -1) {
			return null;
		} else {
			return map;
		}
	}

	public boolean putKeypadMap(char [][] map) {
		if (map.length != KEYPAD_ROWS) {
			return false;
		}
		for (int i=0; i<KEYPAD_ROWS; i++) {
			if (map[i].length != KEYPAD_COLUMNS) {
				return false;
			}
		}
		try {
			FileOutputStream atssetup = new FileOutputStream("/dev/atssetup");
			int outFileSize = 4 + KEYPAD_ROWS*KEYPAD_COLUMNS*4;
			byte [] bytestream = new byte [outFileSize];
			byte [] kpda = "KPDA".getBytes();
			for (int i=0; i<4; i++) {
				bytestream[i] = kpda[i];
			}
			for (int i=0; i<KEYPAD_ROWS; i++) {
				for (int j=0; j<KEYPAD_COLUMNS; i++) {
					bytestream[4+i*j*4+0] = (byte)((map[i][j] & 0xFF000000)>>24);
					bytestream[4+i*j*4+1] = (byte)((map[i][j] & 0x00FF0000)>>16);
					bytestream[4+i*j*4+2] = (byte)((map[i][j] & 0x0000FF00)>> 8);
					bytestream[4+i*j*4+3] = (byte)((map[i][j] & 0x000000FF)>> 0);
				}
			}
            atssetup.write(bytestream);
			return true;
		} catch (Exception ex) {
			return false;
		}
	}

    public boolean restoreDefaultKeymap() {
		try {
			FileOutputStream atssetup = new FileOutputStream("/dev/atssetup");
			byte [] kpda = "KPDA".getBytes();
            atssetup.write(kpda);
		} catch (Exception ex) {
			return false;
		}
		return true;
	}

	public void displayKeypadMap(char [][] map) {
		DecimalFormat df = new DecimalFormat("000");
		System.out.println();
		System.out.print("-");
		for (int j=0; j<KEYPAD_COLUMNS; j++) {
			System.out.print("------");
		}
		System.out.println("");
		for (int i=0; i<KEYPAD_ROWS; i++) {
			System.out.print("|");
			for (int j=0; j<KEYPAD_COLUMNS; j++) {
				System.out.print(" "+df.format(map[i][j])+" |");
			}
			System.out.println();
		}
		System.out.print("-");
		for (int j=0; j<KEYPAD_COLUMNS; j++) {
			System.out.print("------");
		}
		System.out.println();
		for (int i=0; i<KEYPAD_ROWS; i++) {
			System.out.print("| ");
			for (int j=0; j<KEYPAD_COLUMNS; j++) {
				boolean matched = false;
				int k;
				for (k=0; k<keyTriples.length; k++) {
					if (keyTriples[k].keypadValue == map[i][j]) {
						matched = true;
						break;
					}
				}
				if (!matched) {
					System.out.print("_?_");
				} else if (keyTriples[k].unicodeChar == 0) {
					System.out.print("   ");
				} else if (keyTriples[k].unicodeChar == ' ') {
					System.out.print("_ _");
				} else if (keyTriples[k].unicodeChar < ' ') {
					System.out.print("^"+(char)(keyTriples[k].unicodeChar+0x40)+" ");
				} else {
					System.out.print(" "+keyTriples[k].unicodeChar+" ");
				}
				System.out.print(" | ");
			}
			System.out.println();
		}
		for (int j=0; j<KEYPAD_COLUMNS; j++) {
			System.out.print("------");
		}
		System.out.println();
	}

	public String stripTabsAndSpaces(String line) {
		StringBuffer txt = new StringBuffer(line);
		for (int i=0; i<txt.length();) {
			if (txt.charAt(i) == ' ' || txt.charAt(i) == '\t') {
				txt.deleteCharAt(i);
			} else {
				i++;
			}
		}
		return txt.toString();
	}

	public char [][] readKeypadDefFile(String file) {
		try {
			String line;
		    BufferedReader inputStream = new BufferedReader(new FileReader(file));
			char [][] map = new char[KEYPAD_ROWS][KEYPAD_COLUMNS];
			boolean [][] assigned = new boolean[KEYPAD_ROWS][KEYPAD_COLUMNS];
			int row, col, keyCode;
			StringTokenizer tokens;
			while (null != (line = inputStream.readLine())) {
				if (line.indexOf("#") == 0) {
					continue;
				}
				line = stripTabsAndSpaces(line);
				tokens = new StringTokenizer(line, ",=", true);
				if (tokens.countTokens() != 5) {
					Dbg.out("Couldn't parse: " + line);
					continue;
				}
				row = Integer.parseInt(tokens.nextToken());
				if (!tokens.nextToken().equals(",")) {
					Dbg.out("Couldn't parse: " + line);
					continue;
				}
				col = Integer.parseInt(tokens.nextToken());
				if (!tokens.nextToken("=").equals("=")) {
					Dbg.out("Couldn't parse: " + line);
					continue;
				}
				keyCode = Integer.parseInt(tokens.nextToken());
				if (assigned[row][col]) {
					Dbg.out("("+row+","+col+")="+keyCode+" replaced "+map[row][col]);
				}
				map[row][col] = (char)keyCode;
				assigned[row][col] = true;
			}
			for (row=0; row < KEYPAD_ROWS; row++) {
				for (col=0; col < KEYPAD_COLUMNS; col++) {
					if (!assigned[row][col]) {
						Dbg.out("No key code specified for ("+row+","+col+")");
					}
				}
			}
			inputStream.close();
			return map;
	    } catch (Exception ex) {
	        Dbg.out("Exception (" + ex + ") file= " + file);
	        Dbg.pst(ex);
			return null;
		}
	}

	public boolean writeKeypadDefFile(String file, char [][] map) {
		try {
		    BufferedWriter outputStream = new BufferedWriter(new FileWriter(file));

			outputStream.write("#Row and Column information is for reference only.\n");
			outputStream.write("# row, column = keycode number from <linux/input.h>\n");
			for (int row=0; row < KEYPAD_ROWS; row++) {
				for (int col=0; col < KEYPAD_COLUMNS; col++) {
					outputStream.write(row+", "+col+" = "+(int)map[row][col]+"\n");
				}
			}
			outputStream.close();
			return true;
	    } catch (Exception ex) {
	        Dbg.out("Exception (" + ex + ") file= " + file);
	        Dbg.pst(ex);
			return false;
		}
	}
}		
