/*
+====================================================================+
|Global Terminal                                                             |
|  Written by:  Mike Scudder/MicroTools                               |
+====================================================================+
|                                                                    |
|  2007 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/
// This class provides watchdog timers.
//
// A watchdog timer must be hit, repeatedly.  Each hit resets the timer.  If the
// timer ever expires, the system will reboot.  It is used to provide a safety
// net for a program: if the program stops hitting the watchdog, it is presumed
// to have lost its way and the reboot restores the system to proper operation.
//
// This class can be used to create multiple watchdog timers. Each watchdog timer 
// created must be hit repeatedly.  Each watchdog timer has its own timeout. Each
// hit resets the timer to the timeout.
//
// There are two methods provided: hit, and close.  Constructing a watchdog also
// sets its timeout. The timeout parameter is an integer and is in seconds. The
// timeout must be 0 < timeout < 256.
//
// The hit method hits the watchdog timer, meaning it resets the timer to the
// previously set timeout. The watchdog timer should be, in normal operation,
// hit frequently enough that it never expires.
//
// The close method shuts down a watchdog timer; afterwards, it will never cause
// a reboot, and hitting it has no effect.
//
package com.accu_time.util;

import java.io.File;
import java.io.FileOutputStream;

public class SoftwareWatchdog {
	static int watchdogsCreated = 0;
	static File wdtd_pipe = null;
	static FileOutputStream wdtd = null;
	int timeout;
	boolean closed = true;
	byte [] tag;

	final static int WD_REGISTER    = 0x01;
	final static int WD_CHECKIN     = 0x02;
	final static int WD_NEW_TIMEOUT = 0x04;
	final static int WD_UNREGISTER  = 0x08;
	final static int WD_CMDBUFSIZE = 8;
	final static int WD_TAGSIZE = 6;

	final static int POWERD_REBOOT  = 0x11;
	final static int POWERD_HALT    = 0x12;
	final static int POWERD_EXECMD  = 0x13;
	final static int POWERD_CMDSIZE = 4;
	final static int POWERD_CMDSTRSIZE = 1024;

	public SoftwareWatchdog(int timeout) {
		this.timeout = timeout;
		if (timeout <= 0 || timeout >= 256 || watchdogsCreated > 100000) {
			return; // closed flag is still true, this watchdog cannot cause a reboot
		}
		if (wdtd_pipe == null) {
			try {
				wdtd_pipe = new File("/var/wdtd/.wdtdfifo");
				wdtd = new FileOutputStream(wdtd_pipe);
			} catch (Exception ex) {
				System.out.println("Couldn't connect to the wdtd daemon exception=" + ex);
			}
		}
		if (wdtd == null) {
			return; // closed flag is still true, this watchdog cannot cause a reboot
		} else {
			try {
				byte [] buff = new byte [WD_CMDBUFSIZE + POWERD_CMDSIZE + POWERD_CMDSTRSIZE];
				tag = new byte[WD_TAGSIZE];
				int tmp = watchdogsCreated;
				for (int i=0; i<WD_TAGSIZE; i++) {
					tag[i] = (byte)('0' + tmp%10);
					tmp /= 10;
				}
				buff[0] = WD_REGISTER;
				for (int i=0; i<WD_TAGSIZE; i++) {
					buff[i+1] = tag[i];
				}
				buff[7] = (byte)(timeout%256);
				buff[8] = (byte)POWERD_REBOOT;
				for (int i=9; i < buff.length; i++) {
					buff[i] = 0;
				}
				wdtd.write(buff);
				wdtd.flush();
			} catch (Exception ex) {
				System.out.println("Couldn't connect to the wdtd daemon exception=" + ex);
				return; // closed flag is still true, this watchdog cannot cause a reboot
			}
			watchdogsCreated++;
			closed = false;
		}
	}

	public void hit() {
		if (closed) {
			return;
		}
		try {
			byte [] buff = new byte [WD_CMDBUFSIZE];
			buff[0] = WD_CHECKIN;
			for (int i=0; i<WD_TAGSIZE; i++) {
				buff[i+1] = tag[i];
			}
			buff[7] = 0;
			wdtd.write(buff);
			wdtd.flush();
			System.out.print("!");
		} catch (Exception ex) {
			System.out.println("Couldn't write to the wdtd pipe " + wdtd + " exception=" + ex);
		}
	}

	public void close() {
		if (closed) {
			return;
		}
		try {
			byte [] buff = new byte [WD_CMDBUFSIZE];
			buff[0] = WD_UNREGISTER;
			for (int i=0; i<WD_TAGSIZE; i++) {
				buff[i+1] = tag[i];
			}
			buff[7] = 0;
			wdtd.write(buff);
			wdtd.flush();
		} catch (Exception ex) {
			System.out.println("Couldn't write to the wdtd pipe " + wdtd + " exception=" + ex);
		}
		closed = true;
	}

	protected void finalize() throws java.io.IOException {
		if (wdtd != null) {
			wdtd.close();
		}
	}
}
