/*
+====================================================================+
|Global Terminals                                                             |
|  Written by:  Bob Japenga/MicroTools                               |
+====================================================================+
|                                                                    |
|  2007 Accu-Time Systems, Inc.  All rights reserved.               |
|                                                                    |
|====================================================================+
|                                                                    |
|                       DISCLAIMER                                   |
|NO WARRANTIES                                                       |
|Accu-Time Systems expressly disclaims any warranty for the SOFTWARE |
|PRODUCT. The SOFTWARE PRODUCT and any related documentation is      |
|provided "as is" without warranty of any kind, either expressed or  |
|implied, including, without limitation, the implied warranties or   |
|merchantability, fitness for a particular purpose, or noninfringe-  |
|ment. The entire risk arising out of the use or performance of the  |
|SOFTWARE PRODUCT remains with the user.                             |
|                                                                    |
|NO LIABILITY FOR DAMAGES.                                           |
|Under no circumstances is Accu-Time Systems liable for any damages  |
|whatsoever (including, without limitation, damages for loss of busi-|
|ness profits, business interruption, loss of business information,  |
|or any other pecuniary loss) arising out of the use of or inability |
|to use this product.                                                |
|                                                                    |
+====================================================================+
*/
/* TextFileDeviceIO.java */
/* Created by Mike Scudder of MicroTools, Inc. for Accu-time on 20 Feb 2008 */
/* Abstract class for communicating with a device that is controlled
   by reading and writing a text file with a "String = value" format.
 */
// 2011-11-21 MJS Allow for a temporarily blocked text file device

package com.accu_time.util;

import java.io.FileReader;
import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.StringReader;

public class TextFileDeviceIO {
	int tryToReadAtLeast = 16;
	class KnownDevice {
		String devicePath;
		int tryToReadAtLeast;
		KnownDevice(String devicePath, int tryToReadAtLeast) {
			this.devicePath = devicePath;
			this.tryToReadAtLeast = tryToReadAtLeast;
		}
	}
	KnownDevice knownDevices[] = { new KnownDevice("/dev/atsconfig", 640), new KnownDevice("/dev/atsleds", 512)};

	BufferedReader inputStream = null;
	BufferedWriter outputStream = null;
	final String splitter = " = ";
	String[][] inputPairs;
	String[][] outputPairs;
	StringBuffer textOfDevFile = new StringBuffer();
	char [] buff = null;
	int maxObservedDevFileSize = 0;
	String deviceFileName = null;

	public TextFileDeviceIO(String deviceFileName) {
		this.deviceFileName = deviceFileName;
		try {
    	    outputStream = new BufferedWriter(new FileWriter(deviceFileName));
        } catch (Exception ex) {
            Dbg.out("Exception (" + ex + ") attempting to open " + deviceFileName);
            Dbg.pst(ex);
		}
		for (int i=0; i<knownDevices.length; i++) {
			if (knownDevices[i].devicePath.equals(deviceFileName)) {
				tryToReadAtLeast = knownDevices[i].tryToReadAtLeast;
			}
		}
	}

    public void shutdown() {
        if (inputStream != null) {
            try {
                inputStream.close();
            } catch (Exception ex) {
                Dbg.out("Exception " + ex + " closing " + inputStream);
                Dbg.pst(ex);
            };
            /* nullify value to avoid double shutdown */
            inputStream = null;
        }
        if (outputStream != null) {
            try {
                outputStream.close();
            } catch (Exception ex) {
                Dbg.out("Exception " + ex + " closing " + outputStream);
                Dbg.pst(ex);
            };
            /* nullify value to avoid double shutdown */
            outputStream = null;
        }
    }

	private String[] lineToPair(String line) {
		if (line == null) {
			return null;
		}
		if (line.endsWith("\n")) {
			line = line.substring(0, line.length()-"\n".length());
		}
		if (line.endsWith("\r")) {
			line = line.substring(0, line.length()-"\r".length());
		}
		int splitterStartsAt = line.lastIndexOf(splitter);
		if (splitterStartsAt < 1) {
			return null;
		}
		return new String[] {line.substring(0, splitterStartsAt),
							 line.substring(splitterStartsAt + splitter.length())};
	}

	String pairToLine(String parameter, String value) {
		if (parameter == null || value == null || parameter.length() == 0) {
			return null;
		}
		return parameter + splitter + value + " "; // extra space is needed or driver fails to write
	}

	String pairToLine(String [] pair) {
		if (pair == null || pair.length != 2 || pair[0] == null || pair [1] == null || pair[0].length() == 0) {
			return null;
		}
		return pair[0] + splitter + pair[1] + " "; // extra space is needed or driver fails to write
	}

	// returns value if the parameter is matched in the results of a prior read(), else null
	public String getValue(String parameter) {
		if (inputPairs == null) {
			return null;
		}
		for (int i=0; i<inputPairs.length; i++) {
			if (inputPairs[i] != null && inputPairs[i][0].equals(parameter)) {
				return inputPairs[i][1];
			}
		}
		return null;
	}

	public boolean putValue(String parameter, String value) {
		if (parameter == null) {
			return false;
		}
		if (value == null) {
			value = "";
		}
		if (outputPairs == null) {
			outputPairs = new String[][] {{parameter, value}};
		} else {
			String[][] temp = new String[outputPairs.length + 1][];
			for (int i=0; i< outputPairs.length; i++) {
				if (outputPairs[i][0].equals(parameter)) {
					outputPairs[i][1] = value;
					return true;
				}
				temp[i] = outputPairs[i];
			}
			temp[outputPairs.length] = new String[] {parameter, value};
			outputPairs = temp;
		}
		return true;
	}

	public boolean putValue(String [] pair) {
		if (pair == null || pair.length != 2) {
			return false;
		}
		return putValue(pair[0], pair[1]);
	}

	public boolean read() {
		if (buff == null || buff.length < tryToReadAtLeast) {
			buff = new char[tryToReadAtLeast];
		}
		try {
        	inputStream = new BufferedReader(new FileReader(deviceFileName));
        } catch (Exception ex) {
            Dbg.out("Exception (" + ex + ") attempting to open for read " + deviceFileName);
            Dbg.pst(ex);
			inputStream = null;
		}
		if (inputStream == null) {
			return false;
		}
		boolean eof = false;
		textOfDevFile.ensureCapacity(tryToReadAtLeast);
		textOfDevFile.setLength(0);
		try {
			while (!eof) {
				int charsRead = inputStream.read(buff, 0, tryToReadAtLeast);
				if (charsRead == -1) {
					eof = true;
					break;
				} else if (charsRead < buff.length) {
					eof = true;
				}
				textOfDevFile.append(buff, 0, charsRead);
			}
			inputStream.close();
		} catch (Exception ex) {
			Dbg.out("Exception (" + ex + ") attempting to read from " + deviceFileName);
			Dbg.pst(ex);
		}
		if (textOfDevFile.length() > tryToReadAtLeast) {
			tryToReadAtLeast = textOfDevFile.length();
		}
		try {
			BufferedReader br = new BufferedReader(new StringReader(textOfDevFile.toString()));
			int pairs = 0;
//			br.mark(textOfDevFile.length());
			while (br.readLine() != null) {
				pairs++;
			}
			if (pairs == 0) {
				return false;
			}
			inputPairs = new String[pairs][2];
//			br.reset();
			br = new BufferedReader(new StringReader(textOfDevFile.toString()));
			for (int i=0; i < pairs; i++) {
				inputPairs[i] = lineToPair(br.readLine());
			}
			return true;
		} catch (Exception ex) {
            Dbg.out("Exception " + ex);
            Dbg.pst(ex);
			return false;
		}
	}

	public boolean write() {
		if (outputPairs == null) {
			return true; // trivial success
		}
		if (outputStream == null) {
			return false;
		}
		try {
			for (int i=0; i<outputPairs.length; i++) {
				outputStream.write(pairToLine(outputPairs[i]));
				outputStream.newLine();
				outputStream.flush();
			}
			outputStream.flush();
			return true;
		} catch (Exception ex) {
			Dbg.out("Exception (" + ex + ") attempting to write to " + deviceFileName);
			Dbg.pst(ex);
			return false;
		}
	}

	public String readValue(String parameter) {
		if (!read()) {
			return null;
		}
		return getValue(parameter);		
	}

	public boolean writeValue(String parameter, String value) {
		if (!putValue(parameter, value)) {
			return false;
		}
		return write();
	}

	public boolean writeValue(String [] pair) {
		if (pair == null || pair.length != 2) {
			return false;
		}
		return writeValue(pair[0], pair[1]);
	}

	public String[][] getAllPairs() {
		return inputPairs;
	}

	public String[][] readAllPairs() {
		if (!read()) {
			return null;
		}
		return getAllPairs();
	}
}
